//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentReader_h
#define _id3DocumentReader_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentImage.h>
#include <id3Document/id3DocumentTextFieldList.h>
#include <id3Document/id3DocumentTextField.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Reads text field on a document using optical character recognition (OCR).
 */

/**
 * Handle to a DocumentReader object of the Document module.
 */
typedef struct id3DocumentReader *ID3_DOCUMENT_READER;

/**
 * @brief Initializes a DocumentReader object.
 * @param phDocumentReader [in] Handle to the DocumentReader object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentReader_Initialize(ID3_DOCUMENT_READER *phDocumentReader);

/**
 * @brief Releases the memory allocated for a DocumentReader.
 * @param phDocumentReader [in] Handle to the DocumentReader object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentReader_Dispose(ID3_DOCUMENT_READER *phDocumentReader);

/**
 * @brief Checks if the handle object is valid.
 * @param hDocumentReader [in] Handle to the DocumentReader object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentReader_CheckHandle(ID3_DOCUMENT_READER hDocumentReader);

/**
 * Gets the number of threads to use for OCR.
 * Hint: Default value is 1. Allocating more than 1 thread here can increase the speed of the process.
 * @param hDocumentReader [in] Handle to the DocumentReader object.
 * @param threadCount [out] Number of threads to use for OCR.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentReader_GetThreadCount(ID3_DOCUMENT_READER hDocumentReader, int *threadCount);

/**
 * Sets the number of threads to use for OCR.
 * Hint: Default value is 1. Allocating more than 1 thread here can increase the speed of the process.
 * @param hDocumentReader [in] Handle to the DocumentReader object.
 * @param threadCount [in] Number of threads to use for OCR.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentReader_SetThreadCount(ID3_DOCUMENT_READER hDocumentReader, int threadCount);

/**
 * Extracts a crop of the face in a document.
 * @param hDocumentReader [in] Handle to the DocumentReader object.
 * @param hImage [in] The image that contains the document. Must be a realigned BGR image of the document.
 * @param documentName [in] The name of the document to be searched. Must be a key loaded in DocumentLibrary.
 * @param hCroppedFace [out] The cropped image of the face.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentReader_ExtractFace(ID3_DOCUMENT_READER hDocumentReader, ID3_DOCUMENT_IMAGE hImage, const char *documentName, ID3_DOCUMENT_IMAGE hCroppedFace);

/**
 * Reads all the fields in a document.
 * @param hDocumentReader [in] Handle to the DocumentReader object.
 * @param hImage [in] The image that contains the document. Must be a realigned BGR image of the document.
 * @param documentName [in] The name of the document to be searched. Must be a key loaded in DocumentLibrary.
 * @param hReadTexts [out] A list of all the text fields read in the document.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentReader_ReadDocument(ID3_DOCUMENT_READER hDocumentReader, ID3_DOCUMENT_IMAGE hImage, const char *documentName, ID3_DOCUMENT_TEXT_FIELD_LIST hReadTexts);

/**
 * Reads a single field in a document.
 * @param hDocumentReader [in] Handle to the DocumentReader object.
 * @param hImage [in] The image that contains the document. Must be a realigned BGR image of the document.
 * @param documentName [in] The name of the document to be searched. Must be a key loaded in DocumentLibrary.
 * @param fieldName [in] The name of the field to search.
 * @param hReadText [out] Read Text in the specified field.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentReader_ReadField(ID3_DOCUMENT_READER hDocumentReader, ID3_DOCUMENT_IMAGE hImage, const char *documentName, const char *fieldName, ID3_DOCUMENT_TEXT_FIELD hReadText);

#ifdef __cplusplus
}
#endif

#endif
