//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentTextField_h
#define _id3DocumentTextField_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentRectangle.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a text field extracted from a document.
 */

/**
 * Handle to a TextField object of the Document module.
 */
typedef struct id3DocumentTextField *ID3_DOCUMENT_TEXT_FIELD;

/**
 * @brief Initializes a TextField object.
 * @param phTextField [in] Handle to the TextField object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextField_Initialize(ID3_DOCUMENT_TEXT_FIELD *phTextField);

/**
 * @brief Releases the memory allocated for a TextField.
 * @param phTextField [in] Handle to the TextField object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextField_Dispose(ID3_DOCUMENT_TEXT_FIELD *phTextField);

/**
 * @brief Checks if the handle object is valid.
 * @param hTextField [in] Handle to the TextField object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextField_CheckHandle(ID3_DOCUMENT_TEXT_FIELD hTextField);

/**
 * @brief Copies an instance of TextField object to another.
 * @param hTextFieldSrc [in] Handle to the source TextField object.
 * @param hTextFieldDst [in] Handle to the destination TextField object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextField_CopyTo(ID3_DOCUMENT_TEXT_FIELD hTextFieldSrc, ID3_DOCUMENT_TEXT_FIELD hTextFieldDst);

/**
 * Gets the boundaries of the text field in the image.
 * @param hTextField [in] Handle to the TextField object.
 * @param sBounds [out] Boundaries of the text field in the image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextField_GetBounds(ID3_DOCUMENT_TEXT_FIELD hTextField, id3DocumentRectangle *sBounds);

/**
 * Gets the confidence score for the value read from the text field.
 * @param hTextField [in] Handle to the TextField object.
 * @param confidence [out] Confidence score for the value read from the text field.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextField_GetConfidence(ID3_DOCUMENT_TEXT_FIELD hTextField, float *confidence);

/**
 * Gets the name of the text field.
 * @param hTextField [in] Handle to the TextField object.
 * @param name [out] Name of the text field.
 * @param nameSize [in,out] Size of the 'name' buffer. Receives the minimum required buffer size to hold the 'name' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'name' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextField_GetName(ID3_DOCUMENT_TEXT_FIELD hTextField, char *name, int *nameSize);

/**
 * Gets the value of the text field.
 * @param hTextField [in] Handle to the TextField object.
 * @param value [out] Value of the text field.
 * @param valueSize [in,out] Size of the 'value' buffer. Receives the minimum required buffer size to hold the 'value' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'value' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextField_GetValue(ID3_DOCUMENT_TEXT_FIELD hTextField, char *value, int *valueSize);

#ifdef __cplusplus
}
#endif

#endif
