//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentTextFieldList_h
#define _id3DocumentTextFieldList_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentTextField.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a list of text fields extracted from a document.
 */

/**
 * Handle to a TextFieldList object of the Document module.
 */
typedef struct id3DocumentTextFieldList *ID3_DOCUMENT_TEXT_FIELD_LIST;

/**
 * @brief Initializes a TextFieldList object.
 * @param phTextFieldList [in] Handle to the TextFieldList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_Initialize(ID3_DOCUMENT_TEXT_FIELD_LIST *phTextFieldList);

/**
 * @brief Releases the memory allocated for a TextFieldList.
 * @param phTextFieldList [in] Handle to the TextFieldList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_Dispose(ID3_DOCUMENT_TEXT_FIELD_LIST *phTextFieldList);

/**
 * @brief Checks if the handle object is valid.
 * @param hTextFieldList [in] Handle to the TextFieldList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_CheckHandle(ID3_DOCUMENT_TEXT_FIELD_LIST hTextFieldList);

/**
 * @brief Copies an instance of TextFieldList object to another.
 * @param hTextFieldListSrc [in] Handle to the source TextFieldList object.
 * @param hTextFieldListDst [in] Handle to the destination TextFieldList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_CopyTo(ID3_DOCUMENT_TEXT_FIELD_LIST hTextFieldListSrc, ID3_DOCUMENT_TEXT_FIELD_LIST hTextFieldListDst);

/**
 * Clears the TextFieldList object.
 * @param hTextFieldList [in] Handle to the TextFieldList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_Clear(ID3_DOCUMENT_TEXT_FIELD_LIST hTextFieldList);

/**
 * Adds an item to the TextFieldList object.
 * @param hTextFieldList [in] Handle to the TextFieldList object.
 * @param hTextFieldItem [in] TextField item to add.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_Add(ID3_DOCUMENT_TEXT_FIELD_LIST hTextFieldList, ID3_DOCUMENT_TEXT_FIELD hTextFieldItem);

/**
 * Gets an item of the TextFieldList object.
 * @param hTextFieldList [in] Handle to the TextFieldList object.
 * @param index [in] Index of the TextField item to get.
 * @param hTextFieldItem [out] TextField item to get.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_Get(ID3_DOCUMENT_TEXT_FIELD_LIST hTextFieldList, int index, ID3_DOCUMENT_TEXT_FIELD hTextFieldItem);

/**
 * Sets an item of the TextFieldList object.
 * @param hTextFieldList [in] Handle to the TextFieldList object.
 * @param index [in] Index of the TextField item to set.
 * @param hTextFieldItem [in] TextField item to set.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_Set(ID3_DOCUMENT_TEXT_FIELD_LIST hTextFieldList, int index, ID3_DOCUMENT_TEXT_FIELD hTextFieldItem);

/**
 * Gets the number of elements in the TextFieldList object.
 * @param hTextFieldList [in] Handle to the TextFieldList object.
 * @param count [out] Number of elements in the TextField object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_GetCount(ID3_DOCUMENT_TEXT_FIELD_LIST hTextFieldList, int *count);

/**
 * Removes an element of the TextFieldList object.
 * @param hTextFieldList [in] Handle to the TextFieldList object.
 * @param index [in] Index of the TextField item to remove.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_RemoveAt(ID3_DOCUMENT_TEXT_FIELD_LIST hTextFieldList, int index);

/**
 * TextFieldList object.
 * @param hTextFieldList [in] Handle to the TextFieldList object.
 * @param newSize [in] TextField object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentTextFieldList_Resize(ID3_DOCUMENT_TEXT_FIELD_LIST hTextFieldList, int newSize);

#ifdef __cplusplus
}
#endif

#endif
