//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
// ignore_for_file: unused_import
import 'dart:collection';
import 'dart:ffi';
import 'dart:typed_data';
import 'package:ffi/ffi.dart';
import 'face_sdk_generated_bindings.dart';
import 'face_native.dart';

import '../id3_face.dart';

final _finalizer = NativeFinalizer(faceSDK.addresses.id3DetectedFaceList_Dispose.cast());

/// Represents a list of detected faces.
class DetectedFaceList extends Object with IterableMixin<DetectedFace> implements Finalizable {
  /// Native handle.
  late Pointer<Pointer<id3DetectedFaceList>> _pHandle;
  bool _disposable = true;

  /// Gets the native handle.
  /// return The native handle.
  Pointer<id3DetectedFaceList> get handle => _pHandle.value;

  /// Creates a new instance of the DetectedFaceList class.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  DetectedFaceList() {
    _pHandle = calloc();
    try {
      var err = faceSDK.id3DetectedFaceList_Initialize(_pHandle);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      _finalizer.attach(this, _pHandle.cast(), detach: this);
    } finally {}
  }

  /// Creates a new instance of the DetectedFaceList class.
  ///
  /// param handle     Handle to the DetectedFaceList.
  /// throws FaceException An error has occurred during Face Library execution.
  DetectedFaceList.fromHandle(Pointer<id3DetectedFaceList> handle) {
    _pHandle = calloc();
    _pHandle.value = handle;
    _disposable = false;
  }

  /// Releases all resources used by this DetectedFaceList.
  void dispose() {
    if (_disposable) {
      faceSDK.id3DetectedFaceList_Dispose(_pHandle);
      calloc.free(_pHandle);
      _finalizer.detach(this);
    }
  }

  // Copyable methods

  /// Creates a copy of the DetectedFaceList object.
  ///
  /// return The newly created DetectedFaceList object.
  /// throws FaceException An error has occurred during Face Library execution.
  DetectedFaceList clone() {
    DetectedFaceList clone = DetectedFaceList();
    var err = faceSDK.id3DetectedFaceList_CopyTo(_pHandle.value, clone.handle);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
    return clone;
  }

  /// Gets or sets the element at the specified index.
  operator [](int key) => get(key);
  operator []=(int key, DetectedFace item) => set(key, item);

  /// Gets the number of elements contained in the list or dictionary.
  int get count => getCount();

  @override
  Iterator<DetectedFace> get iterator => DetectedFaceListIterator(this);

  // Public methods
  /// Clears the DetectedFaceList object.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  void clear() {
    var err = faceSDK.id3DetectedFaceList_Clear(_pHandle.value);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Adds an item to the DetectedFaceList object.
  ///
  /// param detectedFaceItem DetectedFace item to add.
  /// throws FaceException An error has occurred during Face Library execution.
  void add(DetectedFace detectedFaceItem) {
    var err = faceSDK.id3DetectedFaceList_Add(_pHandle.value, detectedFaceItem.handle);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Gets an item of the DetectedFaceList object.
  ///
  /// param index Index of the DetectedFace item to get.
  /// return DetectedFace item to get.
  /// throws FaceException An error has occurred during Face Library execution.
  DetectedFace get(int index) {
    DetectedFace detectedFaceItem = DetectedFace();
    var err = faceSDK.id3DetectedFaceList_Get(_pHandle.value, index, detectedFaceItem.handle);
    if (err != FaceError.success.value) {
      detectedFaceItem.dispose();
      throw FaceException(err);
    }
    return detectedFaceItem;
  }

  /// Sets an item of the DetectedFaceList object.
  ///
  /// param index Index of the DetectedFace item to set.
  /// param detectedFaceItem DetectedFace item to set.
  /// throws FaceException An error has occurred during Face Library execution.
  void set(int index, DetectedFace detectedFaceItem) {
    var err = faceSDK.id3DetectedFaceList_Set(_pHandle.value, index, detectedFaceItem.handle);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Gets the number of elements in the DetectedFaceList object.
  ///
  /// return Number of elements in the DetectedFace object.
  /// throws FaceException An error has occurred during Face Library execution.
  int getCount() {
    Pointer<Int> pCount = calloc();
    try {
      var err = faceSDK.id3DetectedFaceList_GetCount(_pHandle.value, pCount);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      final vCount = pCount.value;
      return vCount;
    } finally {
      calloc.free(pCount);
    }
  }

  /// Removes an element of the DetectedFaceList object.
  ///
  /// param index Index of the DetectedFace item to remove.
  /// throws FaceException An error has occurred during Face Library execution.
  void removeAt(int index) {
    var err = faceSDK.id3DetectedFaceList_RemoveAt(_pHandle.value, index);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// DetectedFaceList object.
  ///
  /// param newSize DetectedFace object.
  /// throws FaceException An error has occurred during Face Library execution.
  void resize(int newSize) {
    var err = faceSDK.id3DetectedFaceList_Resize(_pHandle.value, newSize);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Finds a detected face by ID.
  ///
  /// param id ID of the detected face to find in the list.
  /// return The found detected face.
  /// throws FaceException An error has occurred during Face Library execution.
  DetectedFace findDetectedFace(int id) {
    DetectedFace detectedFace = DetectedFace();
    var err = faceSDK.id3DetectedFaceList_FindDetectedFace(_pHandle.value, id, detectedFace.handle);
    if (err != FaceError.success.value) {
      detectedFace.dispose();
      throw FaceException(err);
    }
    return detectedFace;
  }

  /// Gets the list of IDs in the list.
  ///
  /// return The list of IDs in the list.
  /// throws FaceException An error has occurred during Face Library execution.
  List<int> getIdList() {
    Pointer<Int> pIds = nullptr;
    Pointer<Int> pIdsSize = calloc();
    pIdsSize[0] = -1;
    try {
      var err = faceSDK.id3DetectedFaceList_GetIdList(_pHandle.value, pIds, pIdsSize);
      if (err == FaceError.insufficientBuffer.value) {
        pIds = calloc.allocate(pIdsSize.value);
        err = faceSDK.id3DetectedFaceList_GetIdList(_pHandle.value, pIds, pIdsSize);
      }
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      final vIds = Int32List.fromList(pIds.cast<Int32>().asTypedList(pIdsSize.value));
      return vIds;
    } finally {
      calloc.free(pIds);
      calloc.free(pIdsSize);
    }
  }

  /// Gets the largest face in the list.
  ///
  /// return The largest detected face in the list.
  /// throws FaceException An error has occurred during Face Library execution.
  DetectedFace getLargestFace() {
    DetectedFace largestFace = DetectedFace();
    var err = faceSDK.id3DetectedFaceList_GetLargestFace(_pHandle.value, largestFace.handle);
    if (err != FaceError.success.value) {
      largestFace.dispose();
      throw FaceException(err);
    }
    return largestFace;
  }

  /// Rescales all detected faces in the list.
  ///
  /// param scale Scale factor to apply. Range is (0;+inf(.
  /// throws FaceException An error has occurred during Face Library execution.
  void rescaleAll(double scale) {
    var err = faceSDK.id3DetectedFaceList_RescaleAll(_pHandle.value, scale);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

}

class DetectedFaceListIterator implements Iterator<DetectedFace> {
  DetectedFaceListIterator(this._list) : _count = _list.getCount();
  final DetectedFaceList _list;
  final int _count;
  int _index = -1;

  @override
  DetectedFace get current => _list.get(_index);

  @override
  bool moveNext() {
    _index++;
    return _index < _count;
  }
}
