//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------

/// Enumerates the possible error codes returned by this SDK.
enum FaceError {
	/// No error.
	success,
	/// The given buffer is too small.
	insufficientBuffer,
	/// The requested object does not exist; a 'null' should be returned if possible.
	objectDoesNotExist,
	/// Base error code of the Face module.
	base,
	/// Minimum error code of the Face module.
	minimum,
	/// The handle is invalid.
	invalidHandle,
	/// The parameter is invalid.
	invalidParameter,
	/// The image pixel format is invalid.
	invalidPixelFormat,
	/// The template format is invalid.
	invalidTemplateFormat,
	/// The computed landmarks are invalid and do not allow to process the desired information.
	invalidLandmarks,
	/// The computed interocular distance (IOD) is too small to allow to process the desired information.
	iodTooSmall,
	/// The model could not be found in the given directory path.
	invalidModelPath,
	/// The input image size is invalid for this operation.
	invalidImageSize,
	/// The model which is required for this operation was not loaded.
	unloadedModel,
	/// The inference backend failed to infer, probably due to inconsistent input.
	inferenceBackendError,
	/// The requested processing unit is not available in this build.
	processingUnitUnavailable,
	/// The face indexer object is already initialized.
	indexerAlreadyInitialized,
	/// The requested id already exists in the Face Indexer object.
	indexerIdAlreadyExists,
	/// The data of the template is corrupted and cannot be added to the Face Indexer object.
	indexerInvalidTemplateData,
	/// Maximum template number of the Face Indexer object has been reached.
	indexerFull,
	/// No template with this id has been found in the Face Indexer object.
	indexerIdNotFound,
	/// This index file is not valid and cannot be loaded.
	indexerInvalidIndexFile,
	/// This function is not implemented.
	functionNotImplemented,
}

extension FaceErrorX on FaceError {
	/// Creates an enumeration value from int.
	/// param value The int value.
	/// return The created FaceError.
  	static FaceError fromValue(int value) {
    	switch (value) {
      	case 0:
        	return FaceError.success;
      	case -1:
        	return FaceError.insufficientBuffer;
      	case -2:
        	return FaceError.objectDoesNotExist;
      	case -1000:
        	return FaceError.base;
      	case -1999:
        	return FaceError.minimum;
      	case -1001:
        	return FaceError.invalidHandle;
      	case -1002:
        	return FaceError.invalidParameter;
      	case -1004:
        	return FaceError.invalidPixelFormat;
      	case -1005:
        	return FaceError.invalidTemplateFormat;
      	case -1006:
        	return FaceError.invalidLandmarks;
      	case -1007:
        	return FaceError.iodTooSmall;
      	case -1008:
        	return FaceError.invalidModelPath;
      	case -1010:
        	return FaceError.invalidImageSize;
      	case -1020:
        	return FaceError.unloadedModel;
      	case -1040:
        	return FaceError.inferenceBackendError;
      	case -1041:
        	return FaceError.processingUnitUnavailable;
      	case -1050:
        	return FaceError.indexerAlreadyInitialized;
      	case -1051:
        	return FaceError.indexerIdAlreadyExists;
      	case -1052:
        	return FaceError.indexerInvalidTemplateData;
      	case -1053:
        	return FaceError.indexerFull;
      	case -1054:
        	return FaceError.indexerIdNotFound;
      	case -1055:
        	return FaceError.indexerInvalidIndexFile;
      	case -1056:
        	return FaceError.functionNotImplemented;
	    default:
        	throw Exception("Unable to find the specified enum value for $value");
    	}
  	}

	/// Gets the enumeration value as int.
	/// return The value.
  	int get value {
    	switch (this) {
      	case FaceError.success:
        	return 0;
      	case FaceError.insufficientBuffer:
        	return -1;
      	case FaceError.objectDoesNotExist:
        	return -2;
      	case FaceError.base:
        	return -1000;
      	case FaceError.minimum:
        	return -1999;
      	case FaceError.invalidHandle:
        	return -1001;
      	case FaceError.invalidParameter:
        	return -1002;
      	case FaceError.invalidPixelFormat:
        	return -1004;
      	case FaceError.invalidTemplateFormat:
        	return -1005;
      	case FaceError.invalidLandmarks:
        	return -1006;
      	case FaceError.iodTooSmall:
        	return -1007;
      	case FaceError.invalidModelPath:
        	return -1008;
      	case FaceError.invalidImageSize:
        	return -1010;
      	case FaceError.unloadedModel:
        	return -1020;
      	case FaceError.inferenceBackendError:
        	return -1040;
      	case FaceError.processingUnitUnavailable:
        	return -1041;
      	case FaceError.indexerAlreadyInitialized:
        	return -1050;
      	case FaceError.indexerIdAlreadyExists:
        	return -1051;
      	case FaceError.indexerInvalidTemplateData:
        	return -1052;
      	case FaceError.indexerFull:
        	return -1053;
      	case FaceError.indexerIdNotFound:
        	return -1054;
      	case FaceError.indexerInvalidIndexFile:
        	return -1055;
      	case FaceError.functionNotImplemented:
        	return -1056;
		}
  	}
}
