//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
import 'dart:ffi';
import 'package:ffi/ffi.dart';
import 'face_sdk_generated_bindings.dart';

/// Represents the geometric attributes of a portrait image.
class GeometricAttributes {
  final Pointer<id3FaceGeometricAttributes> _handle;

  static final Finalizer<Pointer<id3FaceGeometricAttributes>> _finalizer = Finalizer((handle) => calloc.free(handle));

  Pointer<id3FaceGeometricAttributes> get handle => _handle;

  id3FaceGeometricAttributes get struct => _handle.ref;

  /// Initializes a new instance of the GeometricAttributes class from its members.
  GeometricAttributes(this._handle);

  /// Create a new instance of the GeometricAttributes.
  GeometricAttributes.create() : _handle = calloc() {
    _finalizer.attach(this, _handle, detach: this);
  }

  /// Create a new instance of the GeometricAttributes from the native struct
  GeometricAttributes.createFromStruct(id3FaceGeometricAttributes value) : _handle = calloc() {
    _handle.ref.HeadImageHeightRatio = value.HeadImageHeightRatio;
    _handle.ref.HeadImageWidthRatio = value.HeadImageWidthRatio;
    _handle.ref.HorizontalPosition = value.HorizontalPosition;
    _handle.ref.Resolution = value.Resolution;
    _handle.ref.VerticalPosition = value.VerticalPosition;
    _finalizer.attach(this, _handle, detach: this);
  }

  void dispose() {
    calloc.free(_handle);
    _finalizer.detach(this);
  }

  // Accessors
  
  /// 
  /// Value indicating if the distance between the base of the chin and the crown is less than 80% of the total height of the image.
  /// Hint: For ICAO compliance, the value must be in the range 60 to 90.
  /// 
  /// throws FaceException An error has occurred during Face Library execution.
  int get headImageHeightRatio => _handle.ref.HeadImageHeightRatio;
  set headImageHeightRatio(int value) => _handle.ref.HeadImageHeightRatio = value;

  /// 
  /// Value indicating if the image width is conform to the (image width / head width) ratio of 7:5.
  /// The head width is defined as the distance between the left and right ears.
  /// Hint: For ICAO compliance, the value must be in the range 50 to 75.
  /// 
  /// throws FaceException An error has occurred during Face Library execution.
  int get headImageWidthRatio => _handle.ref.HeadImageWidthRatio;
  set headImageWidthRatio(int value) => _handle.ref.HeadImageWidthRatio = value;

  /// 
  /// Value indicating if the head is correctly centered horizontally.
  /// Hint: For ICAO compliance, the value must be in the range 45 to 55.
  /// 
  /// throws FaceException An error has occurred during Face Library execution.
  int get horizontalPosition => _handle.ref.HorizontalPosition;
  set horizontalPosition(int value) => _handle.ref.HorizontalPosition = value;

  /// 
  /// Value indicating if the image resolution is sufficient.
  /// Hint: The recommended value is over 90.
  /// 
  /// throws FaceException An error has occurred during Face Library execution.
  int get resolution => _handle.ref.Resolution;
  set resolution(int value) => _handle.ref.Resolution = value;

  /// 
  /// Value indicating if the distance from the bottom edge of the image to the imaginary line passing through the center of the eyes is between 50% - 70% of the total vertical length of the image.
  /// Hint: For ICAO compliance, the value must be in the range 30 to 50.
  /// 
  /// throws FaceException An error has occurred during Face Library execution.
  int get verticalPosition => _handle.ref.VerticalPosition;
  set verticalPosition(int value) => _handle.ref.VerticalPosition = value;


  Map<String, dynamic> toMap() {
    return {
        'headImageHeightRatio': headImageHeightRatio,
        'headImageWidthRatio': headImageWidthRatio,
        'horizontalPosition': horizontalPosition,
        'resolution': resolution,
        'verticalPosition': verticalPosition,
    };
  }
}
