//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
// ignore_for_file: unused_import
import 'dart:collection';
import 'dart:ffi';
import 'dart:typed_data';
import 'package:ffi/ffi.dart';
import 'face_sdk_generated_bindings.dart';
import 'face_native.dart';

import '../id3_face.dart';

final _finalizer = NativeFinalizer(faceSDK.addresses.id3FaceImageList_Dispose.cast());

/// Represents a list of images.
class ImageList extends Object with IterableMixin<Image> implements Finalizable {
  /// Native handle.
  late Pointer<Pointer<id3FaceImageList>> _pHandle;
  bool _disposable = true;

  /// Gets the native handle.
  /// return The native handle.
  Pointer<id3FaceImageList> get handle => _pHandle.value;

  /// Creates a new instance of the ImageList class.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  ImageList() {
    _pHandle = calloc();
    try {
      var err = faceSDK.id3FaceImageList_Initialize(_pHandle);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      _finalizer.attach(this, _pHandle.cast(), detach: this);
    } finally {}
  }

  /// Creates a new instance of the ImageList class.
  ///
  /// param handle     Handle to the ImageList.
  /// throws FaceException An error has occurred during Face Library execution.
  ImageList.fromHandle(Pointer<id3FaceImageList> handle) {
    _pHandle = calloc();
    _pHandle.value = handle;
    _disposable = false;
  }

  /// Releases all resources used by this ImageList.
  void dispose() {
    if (_disposable) {
      faceSDK.id3FaceImageList_Dispose(_pHandle);
      calloc.free(_pHandle);
      _finalizer.detach(this);
    }
  }

  // Copyable methods

  /// Creates a copy of the ImageList object.
  ///
  /// return The newly created ImageList object.
  /// throws FaceException An error has occurred during Face Library execution.
  ImageList clone() {
    ImageList clone = ImageList();
    var err = faceSDK.id3FaceImageList_CopyTo(_pHandle.value, clone.handle);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
    return clone;
  }

  /// Gets or sets the element at the specified index.
  operator [](int key) => get(key);
  operator []=(int key, Image item) => set(key, item);

  /// Gets the number of elements contained in the list or dictionary.
  int get count => getCount();

  @override
  Iterator<Image> get iterator => ImageListIterator(this);

  // Public methods
  /// Clears the ImageList object.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  void clear() {
    var err = faceSDK.id3FaceImageList_Clear(_pHandle.value);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Adds an item to the ImageList object.
  ///
  /// param imageItem Image item to add.
  /// throws FaceException An error has occurred during Face Library execution.
  void add(Image imageItem) {
    var err = faceSDK.id3FaceImageList_Add(_pHandle.value, imageItem.handle);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Gets an item of the ImageList object.
  ///
  /// param index Index of the Image item to get.
  /// return Image item to get.
  /// throws FaceException An error has occurred during Face Library execution.
  Image get(int index) {
    Image imageItem = Image();
    var err = faceSDK.id3FaceImageList_Get(_pHandle.value, index, imageItem.handle);
    if (err != FaceError.success.value) {
      imageItem.dispose();
      throw FaceException(err);
    }
    return imageItem;
  }

  /// Sets an item of the ImageList object.
  ///
  /// param index Index of the Image item to set.
  /// param imageItem Image item to set.
  /// throws FaceException An error has occurred during Face Library execution.
  void set(int index, Image imageItem) {
    var err = faceSDK.id3FaceImageList_Set(_pHandle.value, index, imageItem.handle);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Gets the number of elements in the ImageList object.
  ///
  /// return Number of elements in the Image object.
  /// throws FaceException An error has occurred during Face Library execution.
  int getCount() {
    Pointer<Int> pCount = calloc();
    try {
      var err = faceSDK.id3FaceImageList_GetCount(_pHandle.value, pCount);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      final vCount = pCount.value;
      return vCount;
    } finally {
      calloc.free(pCount);
    }
  }

  /// Removes an element of the ImageList object.
  ///
  /// param index Index of the Image item to remove.
  /// throws FaceException An error has occurred during Face Library execution.
  void removeAt(int index) {
    var err = faceSDK.id3FaceImageList_RemoveAt(_pHandle.value, index);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// ImageList object.
  ///
  /// param newSize Image object.
  /// throws FaceException An error has occurred during Face Library execution.
  void resize(int newSize) {
    var err = faceSDK.id3FaceImageList_Resize(_pHandle.value, newSize);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

}

class ImageListIterator implements Iterator<Image> {
  ImageListIterator(this._list) : _count = _list.getCount();
  final ImageList _list;
  final int _count;
  int _index = -1;

  @override
  Image get current => _list.get(_index);

  @override
  bool moveNext() {
    _index++;
    return _index < _count;
  }
}
