//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------

/// Enumerates the instructions to be given to the user for effective portrait capture.
enum PortraitInstruction {
	/// No instruction given.
	none,
	/// Center your face in the frame.
	centerFace,
	/// Look directly at the camera.
	lookStraight,
	/// Look away.
	lookAway,
	/// Ensure your face is well-lit.
	adjustLightning,
	/// The image contrast is too low.
	lowContrast,
	/// Image colors are not balanced correctly.
	unbalancedColors,
	/// Image is out of focus.
	imageOutOfFocus,
	/// Move closer to the camera.
	moveCloser,
	/// Move back from the camera.
	moveBack,
	/// Move your face lower to align with the camera.
	moveDown,
	/// Move your face higher to align with the camera.
	moveUp,
	/// Move your face to the right.
	moveRight,
	/// Move your face to the left.
	moveLeft,
	/// Stay still in front of the camera.
	stayStill,
	/// Close your mouth.
	closeMouth,
	/// Open your mouth.
	openMouth,
	/// Smile.
	smile,
	/// Ensure a neutral expression.
	ensureNeutralExpression,
	/// Remove any obstacle from your face.
	facePartiallyHidden,
	/// Turn your head to the left.
	turnHeadLeft,
	/// Turn your head to the right.
	turnHeadRight,
	/// Raise your head.
	raiseHead,
	/// Lower your head.
	lowerHead,
	/// Rotate your head in the specified direction.
	rotateHead,
	/// Turn your head slightly.
	turnHeadSlightly,
	/// Remove any head coverings.
	removeHeadCoverings,
	/// Remove your face mask.
	removeFaceMask,
	/// Remove your glasses.
	removeGlasses,
	/// Remove your sunglasses.
	removeSunglasses,
}

extension PortraitInstructionX on PortraitInstruction {
	/// Creates an enumeration value from int.
	/// param value The int value.
	/// return The created PortraitInstruction.
  	static PortraitInstruction fromValue(int value) {
    	switch (value) {
      	case 0:
        	return PortraitInstruction.none;
      	case 1:
        	return PortraitInstruction.centerFace;
      	case 2:
        	return PortraitInstruction.lookStraight;
      	case 3:
        	return PortraitInstruction.lookAway;
      	case 4:
        	return PortraitInstruction.adjustLightning;
      	case 5:
        	return PortraitInstruction.lowContrast;
      	case 6:
        	return PortraitInstruction.unbalancedColors;
      	case 7:
        	return PortraitInstruction.imageOutOfFocus;
      	case 8:
        	return PortraitInstruction.moveCloser;
      	case 9:
        	return PortraitInstruction.moveBack;
      	case 10:
        	return PortraitInstruction.moveDown;
      	case 11:
        	return PortraitInstruction.moveUp;
      	case 12:
        	return PortraitInstruction.moveRight;
      	case 13:
        	return PortraitInstruction.moveLeft;
      	case 14:
        	return PortraitInstruction.stayStill;
      	case 15:
        	return PortraitInstruction.closeMouth;
      	case 16:
        	return PortraitInstruction.openMouth;
      	case 17:
        	return PortraitInstruction.smile;
      	case 18:
        	return PortraitInstruction.ensureNeutralExpression;
      	case 19:
        	return PortraitInstruction.facePartiallyHidden;
      	case 20:
        	return PortraitInstruction.turnHeadLeft;
      	case 21:
        	return PortraitInstruction.turnHeadRight;
      	case 22:
        	return PortraitInstruction.raiseHead;
      	case 23:
        	return PortraitInstruction.lowerHead;
      	case 24:
        	return PortraitInstruction.rotateHead;
      	case 25:
        	return PortraitInstruction.turnHeadSlightly;
      	case 26:
        	return PortraitInstruction.removeHeadCoverings;
      	case 27:
        	return PortraitInstruction.removeFaceMask;
      	case 28:
        	return PortraitInstruction.removeGlasses;
      	case 29:
        	return PortraitInstruction.removeSunglasses;
	    default:
        	throw Exception("Unable to find the specified enum value for $value");
    	}
  	}

	/// Gets the enumeration value as int.
	/// return The value.
  	int get value {
    	switch (this) {
      	case PortraitInstruction.none:
        	return 0;
      	case PortraitInstruction.centerFace:
        	return 1;
      	case PortraitInstruction.lookStraight:
        	return 2;
      	case PortraitInstruction.lookAway:
        	return 3;
      	case PortraitInstruction.adjustLightning:
        	return 4;
      	case PortraitInstruction.lowContrast:
        	return 5;
      	case PortraitInstruction.unbalancedColors:
        	return 6;
      	case PortraitInstruction.imageOutOfFocus:
        	return 7;
      	case PortraitInstruction.moveCloser:
        	return 8;
      	case PortraitInstruction.moveBack:
        	return 9;
      	case PortraitInstruction.moveDown:
        	return 10;
      	case PortraitInstruction.moveUp:
        	return 11;
      	case PortraitInstruction.moveRight:
        	return 12;
      	case PortraitInstruction.moveLeft:
        	return 13;
      	case PortraitInstruction.stayStill:
        	return 14;
      	case PortraitInstruction.closeMouth:
        	return 15;
      	case PortraitInstruction.openMouth:
        	return 16;
      	case PortraitInstruction.smile:
        	return 17;
      	case PortraitInstruction.ensureNeutralExpression:
        	return 18;
      	case PortraitInstruction.facePartiallyHidden:
        	return 19;
      	case PortraitInstruction.turnHeadLeft:
        	return 20;
      	case PortraitInstruction.turnHeadRight:
        	return 21;
      	case PortraitInstruction.raiseHead:
        	return 22;
      	case PortraitInstruction.lowerHead:
        	return 23;
      	case PortraitInstruction.rotateHead:
        	return 24;
      	case PortraitInstruction.turnHeadSlightly:
        	return 25;
      	case PortraitInstruction.removeHeadCoverings:
        	return 26;
      	case PortraitInstruction.removeFaceMask:
        	return 27;
      	case PortraitInstruction.removeGlasses:
        	return 28;
      	case PortraitInstruction.removeSunglasses:
        	return 29;
		}
  	}
}
