//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
// ignore_for_file: unused_import
import 'dart:collection';
import 'dart:convert';
import 'dart:ffi';
import 'dart:typed_data';
import 'package:ffi/ffi.dart';
import 'face_sdk_generated_bindings.dart';
import 'face_native.dart';

import '../id3_face.dart';

final _finalizer = NativeFinalizer(faceSDK.addresses.id3FaceStringList_Dispose.cast());

/// Represents a list of strings.
class StringList extends Object with IterableMixin<String> implements Finalizable {
  /// Native handle.
  late Pointer<Pointer<id3FaceStringList>> _pHandle;
  bool _disposable = true;

  /// Gets the native handle.
  /// return The native handle.
  Pointer<id3FaceStringList> get handle => _pHandle.value;

  /// Creates a new instance of the StringList class.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  StringList() {
    _pHandle = calloc();
    try {
      var err = faceSDK.id3FaceStringList_Initialize(_pHandle);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      _finalizer.attach(this, _pHandle.cast(), detach: this);
    } finally {}
  }

  /// Creates a new instance of the StringList class.
  ///
  /// param handle     Handle to the StringList.
  /// throws FaceException An error has occurred during Face Library execution.
  StringList.fromHandle(Pointer<id3FaceStringList> handle) {
    _pHandle = calloc();
    _pHandle.value = handle;
    _disposable = false;
  }

  /// Releases all resources used by this StringList.
  void dispose() {
    if (_disposable) {
      faceSDK.id3FaceStringList_Dispose(_pHandle);
      calloc.free(_pHandle);
      _finalizer.detach(this);
    }
  }

  /// Gets or sets the element at the specified index.
  operator [](int key) => get(key);
  operator []=(int key, String item) => set(key, item);

  /// Gets the number of elements contained in the list or dictionary.
  int get count => getCount();

  @override
  Iterator<String> get iterator => StringListIterator(this);

  // Public methods
  /// Clears the StringList object.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  void clear() {
    var err = faceSDK.id3FaceStringList_Clear(_pHandle.value);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Adds an item to the StringList object.
  ///
  /// param item  item to add.
  /// throws FaceException An error has occurred during Face Library execution.
  void add(String? item) {
    Pointer<Char>? pItem = item?.toNativeUtf8().cast<Char>();
    try {
      var err = faceSDK.id3FaceStringList_Add(_pHandle.value, pItem ?? nullptr);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
    } finally {
      if (pItem != null) {
        calloc.free(pItem);
      }
    }
  }

  /// Gets an item of the StringList object.
  ///
  /// param index Index of the  item to get.
  /// return  item to get.
  /// throws FaceException An error has occurred during Face Library execution.
  String get(int index) {
    Pointer<Char> pItem = nullptr;
    Pointer<Int> pItemSize = calloc.allocate(1);
    pItemSize[0] = -1;
    try {
      var err = faceSDK.id3FaceStringList_Get(_pHandle.value, index, pItem, pItemSize);
      if (err == FaceError.insufficientBuffer.value) {
        pItem = calloc.allocate(pItemSize.value);
        err = faceSDK.id3FaceStringList_Get(_pHandle.value, index, pItem, pItemSize);
        if (err != FaceError.success.value) {
          throw FaceException(err);
        }
      }
      final vItem = utf8.decode(Uint8List.fromList(pItem.cast<Uint8>().asTypedList(pItemSize.value)));
      return vItem;
    } finally {
      calloc.free(pItem);
      calloc.free(pItemSize);
    }
  }

  /// Sets an item of the StringList object.
  ///
  /// param index Index of the  item to set.
  /// param item  item to set.
  /// throws FaceException An error has occurred during Face Library execution.
  void set(int index, String? item) {
    Pointer<Char>? pItem = item?.toNativeUtf8().cast<Char>();
    try {
      var err = faceSDK.id3FaceStringList_Set(_pHandle.value, index, pItem ?? nullptr);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
    } finally {
      if (pItem != null) {
        calloc.free(pItem);
      }
    }
  }

  /// Gets the number of elements in the StringList object.
  ///
  /// return Number of elements in the  object.
  /// throws FaceException An error has occurred during Face Library execution.
  int getCount() {
    Pointer<Int> pCount = calloc();
    try {
      var err = faceSDK.id3FaceStringList_GetCount(_pHandle.value, pCount);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      final vCount = pCount.value;
      return vCount;
    } finally {
      calloc.free(pCount);
    }
  }

  /// Removes an element of the StringList object.
  ///
  /// param index Index of the  item to remove.
  /// throws FaceException An error has occurred during Face Library execution.
  void removeAt(int index) {
    var err = faceSDK.id3FaceStringList_RemoveAt(_pHandle.value, index);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// StringList object.
  ///
  /// param newSize  object.
  /// throws FaceException An error has occurred during Face Library execution.
  void resize(int newSize) {
    var err = faceSDK.id3FaceStringList_Resize(_pHandle.value, newSize);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

}

class StringListIterator implements Iterator<String> {
  StringListIterator(this._list) : _count = _list.getCount();
  final StringList _list;
  final int _count;
  int _index = -1;

  @override
  String get current => _list.get(_index);

  @override
  bool moveNext() {
    _index++;
    return _index < _count;
  }
}
