//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
// ignore_for_file: unused_import
import 'dart:ffi';
import 'dart:typed_data';
import 'package:ffi/ffi.dart';
import 'face_sdk_generated_bindings.dart';
import 'face_native.dart';

import '../id3_face.dart';

final _finalizer = NativeFinalizer(faceSDK.addresses.id3TrackedFace_Dispose.cast());

/// Represents a tracked face.
class TrackedFace implements Finalizable {
  /// Native handle.
  late Pointer<Pointer<id3TrackedFace>> _pHandle;
  bool _disposable = true;

  /// Gets the native handle.
  /// return The native handle.
  Pointer<id3TrackedFace> get handle => _pHandle.value;

  /// Creates a new instance of the TrackedFace class.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  TrackedFace() {
    _pHandle = calloc();
    try {
      var err = faceSDK.id3TrackedFace_Initialize(_pHandle);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      _finalizer.attach(this, _pHandle.cast(), detach: this);
    } finally {}
  }

  /// Creates a new instance of the TrackedFace class.
  ///
  /// param handle     Handle to the TrackedFace.
  /// throws FaceException An error has occurred during Face Library execution.
  TrackedFace.fromHandle(Pointer<id3TrackedFace> handle) {
    _pHandle = calloc();
    _pHandle.value = handle;
    _disposable = false;
  }

  /// Releases all resources used by this TrackedFace.
  void dispose() {
    if (_disposable) {
      faceSDK.id3TrackedFace_Dispose(_pHandle);
      calloc.free(_pHandle);
      _finalizer.detach(this);
    }
  }

  // Copyable methods

  /// Creates a copy of the TrackedFace object.
  ///
  /// return The newly created TrackedFace object.
  /// throws FaceException An error has occurred during Face Library execution.
  TrackedFace clone() {
    TrackedFace clone = TrackedFace();
    var err = faceSDK.id3TrackedFace_CopyTo(_pHandle.value, clone.handle);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
    return clone;
  }


  ///
  /// Bounds of the detected face.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  Rectangle get bounds => getBounds();
  set bounds(Rectangle value) => setBounds(value);

  ///
  /// Confidence score of the detected face.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  int get detectionScore => getDetectionScore();

  ///
  /// Computed template the tracked face. It aims at tracking an identity over time and is consolidated.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  FaceTemplate get faceTemplate => getFaceTemplate();

  ///
  /// ID of the detected face.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  int get id => getId();
  set id(int value) => setId(value);

  ///
  /// Landmarks (eyes, nose and mouth corners) of the detected face.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  PointList get landmarks => getLandmarks();
  set landmarks(PointList value) => setLandmarks(value);

  ///
  /// Predicted bounds of the tracked face. Those bounds are computed using a Kalman filter which has the effect of making them smooth and robust to false non-detections.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  Rectangle get predictedBounds => getPredictedBounds();

  ///
  /// Status of the tracked face. See FaceTrackingStatus for more details.
  ///
  /// throws FaceException An error has occurred during Face Library execution.
  TrackingStatus get trackingStatus => getTrackingStatus();

  // Public methods
  /// Gets the bounds of the detected face.
  ///
  /// return Bounds of the detected face.
  /// throws FaceException An error has occurred during Face Library execution.
  Rectangle getBounds() {
    Pointer<id3FaceRectangle> pBounds = calloc();
    var err = faceSDK.id3TrackedFace_GetBounds(_pHandle.value, pBounds);
    if (err != FaceError.success.value) {
    	calloc.free(pBounds);
    	throw FaceException(err);
    }
    return Rectangle(pBounds);
  }

  /// Sets the bounds of the detected face.
  ///
  /// param bounds Bounds of the detected face.
  /// throws FaceException An error has occurred during Face Library execution.
  void setBounds(Rectangle bounds) {
    var err = faceSDK.id3TrackedFace_SetBounds(_pHandle.value, bounds.handle);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Gets the confidence score of the detected face.
  ///
  /// return Confidence score of the detected face.
  /// throws FaceException An error has occurred during Face Library execution.
  int getDetectionScore() {
    Pointer<Int> pDetectionScore = calloc();
    try {
      var err = faceSDK.id3TrackedFace_GetDetectionScore(_pHandle.value, pDetectionScore);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      final vDetectionScore = pDetectionScore.value;
      return vDetectionScore;
    } finally {
      calloc.free(pDetectionScore);
    }
  }

  /// Gets the computed template the tracked face. It aims at tracking an identity over time and is consolidated.
  ///
  /// return Computed template the tracked face. It aims at tracking an identity over time and is consolidated.
  /// throws FaceException An error has occurred during Face Library execution.
  FaceTemplate getFaceTemplate() {
    FaceTemplate faceTemplate = FaceTemplate();
    var err = faceSDK.id3TrackedFace_GetFaceTemplate(_pHandle.value, faceTemplate.handle);
    if (err != FaceError.success.value) {
      faceTemplate.dispose();
      throw FaceException(err);
    }
    return faceTemplate;
  }

  /// Gets the ID of the detected face.
  ///
  /// return ID of the detected face.
  /// throws FaceException An error has occurred during Face Library execution.
  int getId() {
    Pointer<Int> pId = calloc();
    try {
      var err = faceSDK.id3TrackedFace_GetId(_pHandle.value, pId);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      final vId = pId.value;
      return vId;
    } finally {
      calloc.free(pId);
    }
  }

  /// Sets the ID of the detected face.
  ///
  /// param id ID of the detected face.
  /// throws FaceException An error has occurred during Face Library execution.
  void setId(int id) {
    var err = faceSDK.id3TrackedFace_SetId(_pHandle.value, id);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Gets the landmarks (eyes, nose and mouth corners) of the detected face.
  ///
  /// return Landmarks (eyes, nose and mouth corners) of the detected face.
  /// throws FaceException An error has occurred during Face Library execution.
  PointList getLandmarks() {
    PointList landmarks = PointList();
    var err = faceSDK.id3TrackedFace_GetLandmarks(_pHandle.value, landmarks.handle);
    if (err != FaceError.success.value) {
      landmarks.dispose();
      throw FaceException(err);
    }
    return landmarks;
  }

  /// Sets the landmarks (eyes, nose and mouth corners) of the detected face.
  ///
  /// param landmarks Landmarks (eyes, nose and mouth corners) of the detected face.
  /// throws FaceException An error has occurred during Face Library execution.
  void setLandmarks(PointList landmarks) {
    var err = faceSDK.id3TrackedFace_SetLandmarks(_pHandle.value, landmarks.handle);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Gets the predicted bounds of the tracked face. Those bounds are computed using a Kalman filter which has the effect of making them smooth and robust to false non-detections.
  ///
  /// return Predicted bounds of the tracked face. Those bounds are computed using a Kalman filter which has the effect of making them smooth and robust to false non-detections.
  /// throws FaceException An error has occurred during Face Library execution.
  Rectangle getPredictedBounds() {
    Pointer<id3FaceRectangle> pPredictedBounds = calloc();
    var err = faceSDK.id3TrackedFace_GetPredictedBounds(_pHandle.value, pPredictedBounds);
    if (err != FaceError.success.value) {
    	calloc.free(pPredictedBounds);
    	throw FaceException(err);
    }
    return Rectangle(pPredictedBounds);
  }

  /// Gets the status of the tracked face. See FaceTrackingStatus for more details.
  ///
  /// return Status of the tracked face. See FaceTrackingStatus for more details.
  /// throws FaceException An error has occurred during Face Library execution.
  TrackingStatus getTrackingStatus() {
    Pointer<Int32> pTrackingStatus = calloc();
    try {
      var err = faceSDK.id3TrackedFace_GetTrackingStatus(_pHandle.value, pTrackingStatus);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      final vTrackingStatus = TrackingStatusX.fromValue(pTrackingStatus.value);
      return vTrackingStatus;
    } finally {
      calloc.free(pTrackingStatus);
    }
  }

  /// Creates a detected face.
  ///
  /// param bounds Bounds of the detected face.
  /// param detectionScore Confidence score of the detected face.
  /// param id ID of the detected face.
  /// param landmarks Landmarks (eyes, nose and mouth corners) of the detected face.
  /// return The newly created tracked face.
  /// throws FaceException An error has occurred during Face Library execution.
  static TrackedFace create(Rectangle bounds, int detectionScore, int id, PointList landmarks) {
    TrackedFace trackedFace = TrackedFace();
    var err = faceSDK.id3TrackedFace_Create(trackedFace.handle, bounds.handle, detectionScore, id, landmarks.handle);
    if (err != FaceError.success.value) {
      trackedFace.dispose();
      throw FaceException(err);
    }
    return trackedFace;
  }

  /// Imports the face object from a buffer.
  ///
  /// param data Buffer to import the face object from.
  /// return The newly created tracked face.
  /// throws FaceException An error has occurred during Face Library execution.
  static TrackedFace fromBuffer(Uint8List? data) {
    TrackedFace trackedFace = TrackedFace();
    Pointer<UnsignedChar>? pData;
    if (data != null) {
    	pData = calloc.allocate<UnsignedChar>(data.length);
    	pData.cast<Uint8>().asTypedList(data.length).setAll(0, data);
    }
    try {
      var err = faceSDK.id3TrackedFace_FromBuffer(trackedFace.handle, pData ?? nullptr, data?.length ?? 0);
      if (err != FaceError.success.value) {
        trackedFace.dispose();
        throw FaceException(err);
      }
      return trackedFace;
    } finally {
      if (pData != null) {
        calloc.free(pData);
      }
    }
  }

  /// Imports the face object from a file.
  ///
  /// param path Path to the file to import the face object from.
  /// return The newly created tracked face.
  /// throws FaceException An error has occurred during Face Library execution.
  static TrackedFace fromFile(String? path) {
    TrackedFace trackedFace = TrackedFace();
    Pointer<Char>? pPath = path?.toNativeUtf8().cast<Char>();
    try {
      var err = faceSDK.id3TrackedFace_FromFile(trackedFace.handle, pPath ?? nullptr);
      if (err != FaceError.success.value) {
        trackedFace.dispose();
        throw FaceException(err);
      }
      return trackedFace;
    } finally {
      if (pPath != null) {
        calloc.free(pPath);
      }
    }
  }

  /// Gets the tracked face as a detected face.
  ///
  /// return The tracked face as a detected face.
  /// throws FaceException An error has occurred during Face Library execution.
  DetectedFace getAsDetectedFace() {
    DetectedFace detectedFace = DetectedFace();
    var err = faceSDK.id3TrackedFace_GetAsDetectedFace(_pHandle.value, detectedFace.handle);
    if (err != FaceError.success.value) {
      detectedFace.dispose();
      throw FaceException(err);
    }
    return detectedFace;
  }

  /// Gets the distance between the detected face and the camera when using a depth map in pixels.
  ///
  /// param depthImage Depth image to process.
  /// return The estimated distance to camera in pixels.
  /// throws FaceException An error has occurred during Face Library execution.
  int getDistanceToCamera(Image depthImage) {
    Pointer<Int> pDistanceToCamera = calloc();
    try {
      var err = faceSDK.id3TrackedFace_GetDistanceToCamera(_pHandle.value, depthImage.handle, pDistanceToCamera);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      final vDistanceToCamera = pDistanceToCamera.value;
      return vDistanceToCamera;
    } finally {
      calloc.free(pDistanceToCamera);
    }
  }

  /// Gets the bounding box around the detected face with specified aspect ratio and specified margin.
  ///
  /// param aspectRatio Aspect ratio of the bounding box. Default recommended value is 1.33 (4/3).
  /// param margin Relative margin around the detected face. Recommended value is 0.5.
  /// return The portrait bounds.
  /// throws FaceException An error has occurred during Face Library execution.
  Rectangle getExpandedBounds(double aspectRatio, double margin) {
    Pointer<id3FaceRectangle> pPortraitBounds = calloc();
    var err = faceSDK.id3TrackedFace_GetExpandedBounds(_pHandle.value, aspectRatio, margin, pPortraitBounds);
    if (err != FaceError.success.value) {
    	calloc.free(pPortraitBounds);
    	throw FaceException(err);
    }
    return Rectangle(pPortraitBounds);
  }

  /// Gets the distance between the eyes (IOD) of the detected face in pixels.
  ///
  /// return The computed interocular distance (IOD) in pixels.
  /// throws FaceException An error has occurred during Face Library execution.
  int getInterocularDistance() {
    Pointer<Int> pIod = calloc();
    try {
      var err = faceSDK.id3TrackedFace_GetInterocularDistance(_pHandle.value, pIod);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      final vIod = pIod.value;
      return vIod;
    } finally {
      calloc.free(pIod);
    }
  }

  /// Gets the bounding box of the detected face for ICAO portrait cropping.
  /// This method shall be used to ensure compliance with the ICAO standard.
  ///
  /// param eyeImageWidthRatio Ratio between eye distance and image width. Must be in the range )0;1(. Default recommended value is 0.25.
  /// param eyeImageHeightRatio Ratio between eye distance to top and image height. Must be in the range )0;1(. Default recommended value is 0.45.
  /// param imageRatio Ratio between image height and image width. Default recommended value is 1.33 (4/3).
  /// return The portrait bounds of the detected face.
  /// throws FaceException An error has occurred during Face Library execution.
  Rectangle getPortraitBounds(double eyeImageWidthRatio, double eyeImageHeightRatio, double imageRatio) {
    Pointer<id3FaceRectangle> pPortraitBounds = calloc();
    var err = faceSDK.id3TrackedFace_GetPortraitBounds(_pHandle.value, eyeImageWidthRatio, eyeImageHeightRatio, imageRatio, pPortraitBounds);
    if (err != FaceError.success.value) {
    	calloc.free(pPortraitBounds);
    	throw FaceException(err);
    }
    return Rectangle(pPortraitBounds);
  }

  /// Rescales the detected face object bounding box and landmarks. This function can be useful if the image was downscaled to speed up detection, then you need to upscale the detected face to fit the source image size.
  ///
  /// param scale The multiplicative rescaling factor to apply to the face object.
  /// throws FaceException An error has occurred during Face Library execution.
  void rescale(double scale) {
    var err = faceSDK.id3TrackedFace_Rescale(_pHandle.value, scale);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Rotates the face object by a given angle in degrees from a given center.
  ///
  /// param angle Angle of the rotation to apply to the face object.
  /// param center Center of the rotation to apply to the face object.
  /// throws FaceException An error has occurred during Face Library execution.
  void rotate(int angle, Point center) {
    var err = faceSDK.id3TrackedFace_Rotate(_pHandle.value, angle, center.handle);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

  /// Exports the face object to a buffer.
  ///
  /// return The buffer to which the face object is exported.
  /// throws FaceException An error has occurred during Face Library execution.
  Uint8List toBuffer() {
    Pointer<UnsignedChar> pData = nullptr;
    Pointer<Int> pDataSize = calloc();
    pDataSize[0] = -1;
    try {
      var err = faceSDK.id3TrackedFace_ToBuffer(_pHandle.value, pData, pDataSize);
      if (err == FaceError.insufficientBuffer.value) {
        pData = calloc.allocate(pDataSize.value);
        err = faceSDK.id3TrackedFace_ToBuffer(_pHandle.value, pData, pDataSize);
      }
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
      final vData = Uint8List.fromList(pData.cast<Uint8>().asTypedList(pDataSize.value));
      return vData;
    } finally {
      calloc.free(pData);
      calloc.free(pDataSize);
    }
  }

  /// Saves the face object to a file.
  ///
  /// param path Path to the file to export the face object to.
  /// throws FaceException An error has occurred during Face Library execution.
  void toFile(String? path) {
    Pointer<Char>? pPath = path?.toNativeUtf8().cast<Char>();
    try {
      var err = faceSDK.id3TrackedFace_ToFile(_pHandle.value, pPath ?? nullptr);
      if (err != FaceError.success.value) {
        throw FaceException(err);
      }
    } finally {
      if (pPath != null) {
        calloc.free(pPath);
      }
    }
  }

  /// Translates the face object.
  ///
  /// param tx Translation to apply to the face object alongside the x-axis.
  /// param ty Translation to apply to the face object alongside the y-axis.
  /// throws FaceException An error has occurred during Face Library execution.
  void translate(int tx, int ty) {
    var err = faceSDK.id3TrackedFace_Translate(_pHandle.value, tx, ty);
    if (err != FaceError.success.value) {
      throw FaceException(err);
    }
  }

}

