//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------

/// Enumerates the possible face-tracking statuses.
enum TrackingStatus {
	/// No status.
	none,
	/// A face was detected and a track is initialized. Once the required number of consecutive detections will be reached (see MinimumTrackedFaceAge in FaceTracker), the track will move on to 'Confirmed' status.
	initialized,
	/// The face was detected and tracked for the required number of consecutive frames, hence the track is confirmed.
	confirmed,
	/// The face was missed for at least one frame. It can come back to 'Confirmed' if it is re-detected or move on to 'Deleted' if it keeps being missed for a given number of consecutive frames (see MinimumTrackedFaceAge in FaceTracker).
	temporaryLost,
	/// The face was missed for too many frames, the track is over. Remark: this state is purely internal.
	deleted,
}

extension TrackingStatusX on TrackingStatus {
	/// Creates an enumeration value from int.
	/// param value The int value.
	/// return The created TrackingStatus.
  	static TrackingStatus fromValue(int value) {
    	switch (value) {
      	case 0:
        	return TrackingStatus.none;
      	case 1:
        	return TrackingStatus.initialized;
      	case 2:
        	return TrackingStatus.confirmed;
      	case 3:
        	return TrackingStatus.temporaryLost;
      	case 4:
        	return TrackingStatus.deleted;
	    default:
        	throw Exception("Unable to find the specified enum value for $value");
    	}
  	}

	/// Gets the enumeration value as int.
	/// return The value.
  	int get value {
    	switch (this) {
      	case TrackingStatus.none:
        	return 0;
      	case TrackingStatus.initialized:
        	return 1;
      	case TrackingStatus.confirmed:
        	return 2;
      	case TrackingStatus.temporaryLost:
        	return 3;
      	case TrackingStatus.deleted:
        	return 4;
		}
  	}
}
