//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FaceAnalyser_h
#define _id3FaceAnalyser_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceImage.h>
#include <id3Face/id3DetectedFace.h>
#include <id3Face/id3FaceAttributes.h>
#include <id3Face/id3FaceBackgroundUniformity.h>
#include <id3Face/id3FaceExpression.h>
#include <id3Face/id3FacePointList.h>
#include <id3Face/id3FaceEyeGaze.h>
#include <id3Face/id3FaceGeometricAttributes.h>
#include <id3Face/id3FaceGlassesAttributes.h>
#include <id3Face/id3FacePhotographicAttributes.h>
#include <id3Face/id3FacePose.h>
#include <id3Face/id3FaceOcclusionScores.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Provides methods for analysing the characteristics of portrait image.
 */

/**
 * Handle to a FaceAnalyser object of the Face module.
 */
typedef struct id3FaceAnalyser *ID3_FACE_ANALYSER;

/**
 * @brief Initializes a FaceAnalyser object.
 * @param phFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_Initialize(ID3_FACE_ANALYSER *phFaceAnalyser);

/**
 * @brief Releases the memory allocated for a FaceAnalyser.
 * @param phFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_Dispose(ID3_FACE_ANALYSER *phFaceAnalyser);

/**
 * @brief Checks if the handle object is valid.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_CheckHandle(ID3_FACE_ANALYSER hFaceAnalyser);

/**
 * Gets the sensibility of the expression classifier.
 * The higher the value, the more sensitive the algorithm will be, meaning that it will be less likely to estimate neutral expression out of the detected face.
 * Hint: Default value is 60.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param expressionSensibility [out] Sensibility of the expression classifier.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_GetExpressionSensibility(ID3_FACE_ANALYSER hFaceAnalyser, int *expressionSensibility);

/**
 * Sets the sensibility of the expression classifier.
 * The higher the value, the more sensitive the algorithm will be, meaning that it will be less likely to estimate neutral expression out of the detected face.
 * Hint: Default value is 60.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param expressionSensibility [in] Sensibility of the expression classifier.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_SetExpressionSensibility(ID3_FACE_ANALYSER hFaceAnalyser, int expressionSensibility);

/**
 * Gets the sensibility of the over-exposure classifier, from 0 to 255.
 * The lower the value, the more sensitive the algorithm will be, meaning that it will be more likely to estimate over-exposed face images.
 * Hint: Default value is 188.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param overExposureSensibility [out] Sensibility of the over-exposure classifier, from 0 to 255.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_GetOverExposureSensibility(ID3_FACE_ANALYSER hFaceAnalyser, int *overExposureSensibility);

/**
 * Sets the sensibility of the over-exposure classifier, from 0 to 255.
 * The lower the value, the more sensitive the algorithm will be, meaning that it will be more likely to estimate over-exposed face images.
 * Hint: Default value is 188.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param overExposureSensibility [in] Sensibility of the over-exposure classifier, from 0 to 255.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_SetOverExposureSensibility(ID3_FACE_ANALYSER hFaceAnalyser, int overExposureSensibility);

/**
 * Gets the number of threads to be used for face analysis.
 * Hint: Default value is 1. Allocating more than one thread can increase the speed of the process.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param threadCount [out] Number of threads to be used for face analysis.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_GetThreadCount(ID3_FACE_ANALYSER hFaceAnalyser, int *threadCount);

/**
 * Sets the number of threads to be used for face analysis.
 * Hint: Default value is 1. Allocating more than one thread can increase the speed of the process.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param threadCount [in] Number of threads to be used for face analysis.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_SetThreadCount(ID3_FACE_ANALYSER hFaceAnalyser, int threadCount);

/**
 * Gets the sensibility of the under-exposure classifier, from 0 to 255.
 * The higher the value, the more sensitive the algorithm will be, meaning that it will be more likely to estimate under-exposed face images.
 * Hint: Default value is 66.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param underExposureSensibility [out] Sensibility of the under-exposure classifier, from 0 to 255.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_GetUnderExposureSensibility(ID3_FACE_ANALYSER hFaceAnalyser, int *underExposureSensibility);

/**
 * Sets the sensibility of the under-exposure classifier, from 0 to 255.
 * The higher the value, the more sensitive the algorithm will be, meaning that it will be more likely to estimate under-exposed face images.
 * Hint: Default value is 66.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param underExposureSensibility [in] Sensibility of the under-exposure classifier, from 0 to 255.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_SetUnderExposureSensibility(ID3_FACE_ANALYSER hFaceAnalyser, int underExposureSensibility);

/**
 * Applies an alpha mask to suppress the background and returns a 32-bit BGRA image.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hMask [in] Mask of segmented face.
 * @param hSegmentedFace [out] The output image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ApplyAlphaMask(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_FACE_IMAGE hMask, ID3_FACE_IMAGE hSegmentedFace);

/**
 * Applies a mask to replace the background with the specified color and returns a 24-bit BGR image.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hMask [in] Mask to be applied. Must be a 8-bit greyscale image of same size as the input image.
 * @param red [in] Green channel of the background color. Must be a value from 0 to 255.
 * @param green [in] Red channel of the background color. Must be a value from 0 to 255.
 * @param blue [in] Blue channel of the background color. Must be a value from 0 to 255.
 * @param hSegmentedFace [out] The output image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ApplyMask(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_FACE_IMAGE hMask, int red, int green, int blue, ID3_FACE_IMAGE hSegmentedFace);

/**
 * Verifies if an image is colorized or grayscale.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param isColorized [out] The boolean decision: True if colorized, False if grayscale.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_CheckColorizedImage(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, bool *isColorized);

/**
 * Computes the age of a detected face.
 * Important: Loading the ``FaceAgeEstimator`` model is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param age [out] The estimated age of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeAge(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, int *age);

/**
 * Computes the attributes of a detected face.
 * Important: Loading the model ``FaceAttributesClassifier`` is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param sAttributes [out] The estimated attributes of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeAttributes(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, id3FaceAttributes *sAttributes);

/**
 * Computes the background uniformity behind a detected face.
 * This function must be used for a portrait image with only one face in it. A high uniformity score means that the background is uniform.
 * Important: Loading the model ``FaceBackgroundUniformity`` is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param sBackgroundUniformity [out] The estimated background uniformity. Both color and texture scores are in the range [0:100]. The minimum recommended thresholds are respectively 80 for color and 80 for texture.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeBackgroundUniformity(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, id3FaceBackgroundUniformity *sBackgroundUniformity);

/**
 * Computes the expression of a detected face.
 * Important: Loading the ``FaceExpressionClassifier`` model is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param eExpression [out] The estimated expression of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeExpression(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, id3FaceExpression *eExpression);

/**
 * Computes the eye gaze of a detected face.
 * Important: Loading the ``EyeGazeEstimator`` model is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param hLandmarks [in] Estimated landmarks of the detected face. Must be computed with the face analyser.
 * @param sEyeGaze [out] The estimated gaze of left and right eyes of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeEyeGaze(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, ID3_FACE_POINT_LIST hLandmarks, id3FaceEyeGaze *sEyeGaze);

/**
 * Computes the eye openness of a detected face.
 * The minimum recommanded value for is 90.
 * Important: Loading the ``EyeOpennessDetector`` model is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param hLandmarks [in] Estimated landmarks of the detected face. Must be computed with the face analyser.
 * @param eyeOpennessScores [out] The estimated openness scores of left and right eyes of the detected face in this order.
 * @param eyeOpennessScoresSize [in,out] Size of the 'eyeOpennessScores' buffer. Receives the number of elements in the 'eyeOpennessScores' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeEyeOpenness(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, ID3_FACE_POINT_LIST hLandmarks, int *eyeOpennessScores, int *eyeOpennessScoresSize);

/**
 * Computes the eye redness of a detected face.
 * The maximum recommanded value for an ICAO-compliant portrait is 10.
 * Important: Loading the ``EyeRednessDetector`` model is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param hLandmarks [in] Estimated landmarks of the detected face. Must be computed with the face analyser.
 * @param eyeRednessScores [out] The estimated redness scores of left and right eyes of the detected face in this order.
 * @param eyeRednessScoresSize [in,out] Size of the 'eyeRednessScores' buffer. Receives the number of elements in the 'eyeRednessScores' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeEyeRedness(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, ID3_FACE_POINT_LIST hLandmarks, int *eyeRednessScores, int *eyeRednessScoresSize);

/**
 * Computes the geometric attributes of a detected face. This function must be used for a portrait image with only one face in it.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param hLandmarks [in] Estimated landmarks of the detected face. Must be computed with the face analyser.
 * @param sGeometricAttributes [out] The estimated geometric attributes of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeGeometricAttributes(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, ID3_FACE_POINT_LIST hLandmarks, id3FaceGeometricAttributes *sGeometricAttributes);

/**
 * Computes the glasses attributes of a detected face. This function must be used for a portrait image with only one face in it.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param hLandmarks [in] Estimated landmarks of the detected face. Must be computed with the face analyser.
 * @param sGlassesAttributes [out] The estimated glasses attributes of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeGlassesAttributes(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, ID3_FACE_POINT_LIST hLandmarks, id3FaceGlassesAttributes *sGlassesAttributes);

/**
 * Computes 68 landmarks of a detected face.
 * Important: Loading the ``FaceLandmarksEstimator`` model is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param hLandmarks [out] The estimated landmarks of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeLandmarks(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, ID3_FACE_POINT_LIST hLandmarks);

/**
 * Measures the image noise.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param noiseScore [out] The estimated noise score. 0 means no noise, 100 means no signal.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputeNoise(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, int *noiseScore);

/**
 * Computes the photographic attributes of a detected face. This function must be used for a portrait image with only one face in it.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param hLandmarks [in] Estimated landmarks of the detected face. Must be computed with the face analyser.
 * @param sPhotographicAttributes [out] The estimated photographic attributes of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputePhotographicAttributes(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, ID3_FACE_POINT_LIST hLandmarks, id3FacePhotographicAttributes *sPhotographicAttributes);

/**
 * Computes the pose of a detected face.
 * Important: Loading the ``FacePoseEstimator`` model is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hDetectedFace [in] Detected face to process.
 * @param sPose [out] The estimated pose of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputePose(ID3_FACE_ANALYSER hFaceAnalyser, ID3_DETECTED_FACE hDetectedFace, id3FacePose *sPose);

/**
 * Computes the pose of a detected face using the specified 68-point landmarks.
 * Loading the model ``FacePoseEstimator1A`` is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hDetectedFace [in] Detected face to process.
 * @param hLandmarks [in] Estimated landmarks of the detected face. Must be computed with the face analyser.
 * @param sPose [out] The estimated pose of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_ComputePoseWithLandmarks(ID3_FACE_ANALYSER hFaceAnalyser, ID3_DETECTED_FACE hDetectedFace, ID3_FACE_POINT_LIST hLandmarks, id3FacePose *sPose);

/**
 * Detects the presence of a face mask on a detected face.
 * A high score means that there is a high chance that the person is wearing a mask.
 * The minimum recommended threshold is 15.
 * Important: Loading the ``FaceMaskClassifier``, model is required to use this function.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param score [out] The face mask presence score of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_DetectFaceMask(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, int *score);

/**
 * Detects occlusions on the subject's face.
 * Important: This function requires the ``FaceOcclusionDetector`` model to be loaded.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param sScores [out] The occlusion scores of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_DetectOcclusions(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, id3FaceOcclusionScores *sScores);

/**
 * Upscale and enhance the image of a face.
 * Important: This methods requires the ``CompressionArtifactRemover`` model to be loaded.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hEnhancedImage [out] The enhanced face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_RemoveCompressionArtifacts(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_FACE_IMAGE hEnhancedImage);

/**
 * Computes face segmentation mask for background removal.
 * Important: This methods requires the ``FaceBackgroundSegmenter`` model to be loaded.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hSegmentationMask [out] The mask of segmented face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_SegmentBackground(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_FACE_IMAGE hSegmentationMask);

/**
 * Computes a segmentation map of a face.
 * Important: This methods requires the ``FaceSegmenter`` model to be loaded.
 * @param hFaceAnalyser [in] Handle to the FaceAnalyser object.
 * @param hImage [in] Source image to process.
 * @param hSegmentationMap [out] The map of segmented face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceAnalyser_SegmentFace(ID3_FACE_ANALYSER hFaceAnalyser, ID3_FACE_IMAGE hImage, ID3_FACE_IMAGE hSegmentationMap);

#ifdef __cplusplus
}
#endif

#endif
