//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FaceDetector_h
#define _id3FaceDetector_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceModel.h>
#include <id3Face/id3FaceProcessingUnit.h>
#include <id3Face/id3FaceImage.h>
#include <id3Face/id3DetectedFaceList.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Detects human faces in images.
 */

/**
 * Handle to a FaceDetector object of the Face module.
 */
typedef struct id3FaceDetector *ID3_FACE_DETECTOR;

/**
 * @brief Initializes a FaceDetector object.
 * @param phFaceDetector [in] Handle to the FaceDetector object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_Initialize(ID3_FACE_DETECTOR *phFaceDetector);

/**
 * @brief Releases the memory allocated for a FaceDetector.
 * @param phFaceDetector [in] Handle to the FaceDetector object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_Dispose(ID3_FACE_DETECTOR *phFaceDetector);

/**
 * @brief Checks if the handle object is valid.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_CheckHandle(ID3_FACE_DETECTOR hFaceDetector);

/**
 * Gets the confidence threshold, in the range [0;100].
 * Hint: Default value is 50.
 * Note: Setting a high threshold reduces false detection but can increase the number of undetected faces.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param confidenceThreshold [out] Confidence threshold, in the range [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_GetConfidenceThreshold(ID3_FACE_DETECTOR hFaceDetector, int *confidenceThreshold);

/**
 * Sets the confidence threshold, in the range [0;100].
 * Hint: Default value is 50.
 * Note: Setting a high threshold reduces false detection but can increase the number of undetected faces.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param confidenceThreshold [in] Confidence threshold, in the range [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_SetConfidenceThreshold(ID3_FACE_DETECTOR hFaceDetector, int confidenceThreshold);

/**
 * Gets the model used to detect and track faces.
 * Hint: Default value is FaceDetector4A which is the best available algorithm in this SDK in terms of accuracy. Some better accuracy/speed balances can be found by choosing another model.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param eModel [out] Model used to detect and track faces.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_GetModel(ID3_FACE_DETECTOR hFaceDetector, id3FaceModel *eModel);

/**
 * Sets the model used to detect and track faces.
 * Hint: Default value is FaceDetector4A which is the best available algorithm in this SDK in terms of accuracy. Some better accuracy/speed balances can be found by choosing another model.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param eModel [in] Model used to detect and track faces.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_SetModel(ID3_FACE_DETECTOR hFaceDetector, id3FaceModel eModel);

/**
 * Gets the non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range is [0;100].
 * Hint: Default value is 40.
 * Note: Setting a high threshold allows to detect more overlapping faces which can be useful in a multi-face scenario. On the contrary, in a portrait scenario, a low NMS IOU threshold should be preferred.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param nmsIouThreshold [out] Non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range is [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_GetNmsIouThreshold(ID3_FACE_DETECTOR hFaceDetector, int *nmsIouThreshold);

/**
 * Sets the non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range is [0;100].
 * Hint: Default value is 40.
 * Note: Setting a high threshold allows to detect more overlapping faces which can be useful in a multi-face scenario. On the contrary, in a portrait scenario, a low NMS IOU threshold should be preferred.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param nmsIouThreshold [in] Non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range is [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_SetNmsIouThreshold(ID3_FACE_DETECTOR hFaceDetector, int nmsIouThreshold);

/**
 * Gets the processing unit where to run the detection process.
 * Hint: Default value is CPU.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param eProcessingUnit [out] Processing unit where to run the detection process.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_GetProcessingUnit(ID3_FACE_DETECTOR hFaceDetector, id3FaceProcessingUnit *eProcessingUnit);

/**
 * Sets the processing unit where to run the detection process.
 * Hint: Default value is CPU.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param eProcessingUnit [in] Processing unit where to run the detection process.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_SetProcessingUnit(ID3_FACE_DETECTOR hFaceDetector, id3FaceProcessingUnit eProcessingUnit);

/**
 * Gets the number of threads to be used for face detection.
 * Hint: Default value is 1. Allocating more than one thread can increase the speed of the process.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param threadCount [out] Number of threads to be used for face detection.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_GetThreadCount(ID3_FACE_DETECTOR hFaceDetector, int *threadCount);

/**
 * Sets the number of threads to be used for face detection.
 * Hint: Default value is 1. Allocating more than one thread can increase the speed of the process.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param threadCount [in] Number of threads to be used for face detection.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_SetThreadCount(ID3_FACE_DETECTOR hFaceDetector, int threadCount);

/**
 * Detects faces in an image and store their info in a DetectedFaceList object.
 * Note: The algorithm searches for faces in the range [16px;512px]. If the image is too large to fit this range, one must resize it before the detection process.
 * Important: Loading a ``FaceDetector`` model is required to use this function.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFaceList [out] The list of detected faces.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_DetectFaces(ID3_FACE_DETECTOR hFaceDetector, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE_LIST hDetectedFaceList);

/**
 * Tracks faces in an image and update their info in a DetectedFaceList object.
 * In a realtime process, one must use this function instead of DetectFaces() to keep the face IDs stable in time.
 * Note: The algorithm searches for faces in the range [16px;512px]. If the image is too large to fit this range, one must resize it before the detection process.
 * Important: Loading a ``FaceDetector`` model is required to use this function.
 * Warning: This function is deprecated: use the FaceTracker module instead.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFaceList [in] List of detected faces.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_TrackFaces(ID3_FACE_DETECTOR hFaceDetector, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE_LIST hDetectedFaceList);

/**
 * Gets the face detector module ready to work on a specific image size by initializing all its internal memory layout.
 * This function can be called after setting all the parameters of the face detector module if the image size is fixed, for example using frames from a video stream.
 * Calling this function is not mandatory as the internal memory layout is automatically initialized in the first call to detectFaces() or trackFaces() if missing.
 * Important: Loading a ``FaceDetector`` model is required to use this function.
 * @param hFaceDetector [in] Handle to the FaceDetector object.
 * @param imageWidth [in] Width of the expected image size.
 * @param imageHeight [in] Height of expected image size.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceDetector_WarmUp(ID3_FACE_DETECTOR hFaceDetector, int imageWidth, int imageHeight);

#ifdef __cplusplus
}
#endif

#endif
