//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FaceLibrary_h
#define _id3FaceLibrary_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceModel.h>
#include <id3Face/id3FaceLibVersion.h>
#include <id3Face/id3FaceProcessingUnit.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * The FaceLibrary class serves as the entry point to the id3 Face SDK. It provides essential methods for initializing the SDK, managing face models, and accessing version information. This class must be utilized to leverage the facial recognition capabilities offered by the SDK.
 */

/**
 * Retrieves the model file name which is needed in the LoadModel function.
 * @param eModel [in] The Face Model to look for.
 * @param fileName [out] The expected file name
 * @param fileNameSize [in,out] Size of the 'fileName' buffer. Receives the minimum required buffer size to hold the 'fileName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'fileName' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLibrary_GetModelFileName(id3FaceModel eModel, char *fileName, int *fileNameSize);

/**
 * Retrieves the library version as a `X.Y.Z` formatted string.
 * @param libraryVersion [out] A string that identifies the library version.
 * @param libraryVersionSize [in,out] Size of the 'libraryVersion' buffer. Receives the minimum required buffer size to hold the 'libraryVersion' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'libraryVersion' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLibrary_GetVersion(char *libraryVersion, int *libraryVersionSize);

/**
 * Retrieves the library version as a structure.
 * @param sVersion [out] The version.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLibrary_GetVersionEx(id3FaceLibVersion *sVersion);

/**
 * Loads a specified AI model into memory from the specified directory.
 * @param modelPath [in] The path to directory containing the AI model files.
 * @param eFaceModel [in] The AI model to be loaded.
 * @param eProcessingUnit [in] The processing unit to be used.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLibrary_LoadModel(const char *modelPath, id3FaceModel eFaceModel, id3FaceProcessingUnit eProcessingUnit);

/**
 * Loads a model into memory from the specified buffer.
 * @param modelBuffer [in] A buffer containing the AI model to be loaded.
 * @param modelBufferSize [in] Size of the 'modelBuffer' buffer.
 * @param eFaceModel [in] The AI model to be loaded.
 * @param eProcessingUnit [in] The processing unit to be used.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLibrary_LoadModelBuffer(unsigned char *modelBuffer, int modelBufferSize, id3FaceModel eFaceModel, id3FaceProcessingUnit eProcessingUnit);

/**
 * Unloads a model from memory.
 * @param eFaceModel [in] The AI model to be unloaded.
 * @param eProcessingUnit [in] The processing unit used for the model.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLibrary_UnloadModel(id3FaceModel eFaceModel, id3FaceProcessingUnit eProcessingUnit);

#ifdef __cplusplus
}
#endif

#endif
