//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FaceLicense_h
#define _id3FaceLicense_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceLicenseHardwareCodeType.h>
#include <id3Face/id3FaceLicenseType.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Handles the license to use this SDK.
 */

/**
 * Activates a license using customer credentials and a product reference and saves the license to a file.
 * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
 * @param login [in] The id3 account login.
 * @param password [in] The id3 account password.
 * @param productReference [in] The requested product reference.
 * @param commentary [in] Commentary associated to this license activation.
 * @param path [in] Path to the retrieved license file.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_Activate(const char *hardwareCode, const char *login, const char *password, const char *productReference, const char *commentary, const char *path);

/**
 * Activates a license file using an activation key and saves the license to a file.
 * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
 * @param activationKey [in] The activation key.
 * @param commentary [in] Commentary associated to this license activation, generally the host name.
 * @param path [in] Path to the retrieved license file.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_ActivateActivationKey(const char *hardwareCode, const char *activationKey, const char *commentary, const char *path);

/**
 * Activates a license using an activation key and returns the license in a data buffer.
 * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
 * @param activationKey [in] The id3 activation key.
 * @param commentary [in] Commentary associated to this license activation.
 * @param licenseFileBuffer [out] The license data buffer.
 * @param licenseFileBufferSize [in,out] Size of the 'licenseFileBuffer' buffer. Receives the number of elements in the 'licenseFileBuffer' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_ActivateActivationKeyBuffer(const char *hardwareCode, const char *activationKey, const char *commentary, unsigned char *licenseFileBuffer, int *licenseFileBufferSize);

/**
 * Retrieves a license file buffer using customer credentials and a product reference and returns the license in a data buffer.
 * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
 * @param login [in] The customer login.
 * @param password [in] The customer password.
 * @param productReference [in] The requested product reference.
 * @param commentary [in] Commentary associated to this license activation, generally the host name.
 * @param licenseFileBuffer [out] The license data.
 * @param licenseFileBufferSize [in,out] Size of the 'licenseFileBuffer' buffer. Receives the number of elements in the 'licenseFileBuffer' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_ActivateBuffer(const char *hardwareCode, const char *login, const char *password, const char *productReference, const char *commentary, unsigned char *licenseFileBuffer, int *licenseFileBufferSize);

/**
 * Activates a license using a serial key and saves the license to a file.
 * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
 * @param serialKey [in] The serial key to activate.
 * @param commentary [in] Commentary associated to this license activation, generally the host name.
 * @param path [in] Path to the license file.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_ActivateSerialKey(const char *hardwareCode, const char *serialKey, const char *commentary, const char *path);

/**
 * Activates a license using a serial key and returns the license in a data buffer.
 * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
 * @param serialKey [in] The id3 serial key.
 * @param commentary [in] Commentary associated to this license activation.
 * @param licenseFileBuffer [out] The license data buffer.
 * @param licenseFileBufferSize [in,out] Size of the 'licenseFileBuffer' buffer. Receives the number of elements in the 'licenseFileBuffer' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_ActivateSerialKeyBuffer(const char *hardwareCode, const char *serialKey, const char *commentary, unsigned char *licenseFileBuffer, int *licenseFileBufferSize);

/**
 * Checks a license from a file.
 * This function does two things:
 * - First it loads the license from a file using the given path
 * - Then it checks the validity of the license regarding the host it is called on (Windows, Linux, Android ...)
 * Important: Calling this function is required to get access to the license member getters.
 * Note: The optional parameter is required on Android and must be the JNIEnv* pointer casted as a void*. It can be set as NULL for other devices.
 * @param licensePath [in] Absolute or relative path to the license file.
 * @param optionalParameter [in] Only used on Android: JNIEnv* pointer casted as a void*. Ignored for other devices.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_CheckLicense(const char *licensePath, void *optionalParameter);

/**
 * Checks a license from a buffer.
 * This function does two things:
 * - First it loads the license from a buffer
 * - Then it checks the validity of the license regarding the host it is called on (Windows, Linux, Android ...)
 * Important: Calling one of the license checking function is required to get access to all the license member getters.
 * Note: The optional parameter is required on Android and must be the JNIEnv* pointer casted as a void*. It can be set as NULL for other devices.
 * @param licenseData [in] A buffer containing the license file data.
 * @param licenseDataSize [in] Size of the 'licenseData' buffer.
 * @param optionalParameter [in] Only used on Android: JNIEnv* pointer casted as a void*. Ignored for other devices.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_CheckLicenseBuffer(unsigned char *licenseData, int licenseDataSize, void *optionalParameter);

/**
 * Checks the license using a buffer containing the license data and a card-based verification process.
 * It outputs a challenge response for further verification.
 * @param licenseData [in] A buffer containing the license file data.
 * @param licenseDataSize [in] Size of the 'licenseData' buffer.
 * @param challengeR1 [out] An array to hold the challenge response generated by the host.
 * @param challengeR1Size [in,out] Size of the 'challengeR1' buffer. Receives the number of elements in the 'challengeR1' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_CheckLicenseBufferWithCard(unsigned char *licenseData, int licenseDataSize, unsigned char *challengeR1, int *challengeR1Size);

/**
 * Checks the license by using a card-based verification process.
 * It takes the path to the license file and outputs a challenge response for further verification.
 * @param licensePath [in] The file path to the license that needs to be checked.
 * @param challengeR1 [out] An array to hold the challenge response generated by the host.
 * @param challengeR1Size [in,out] Size of the 'challengeR1' buffer. Receives the number of elements in the 'challengeR1' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_CheckLicenseWithCard(const char *licensePath, unsigned char *challengeR1, int *challengeR1Size);

/**
 * Checks if a module is present in the previously loaded license.
 * @param moduleName [in] The module name.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_CheckModule(const char *moduleName);

/**
 * Finalizes the license check process using the provided cryptogram.
 * It ensures the integrity and authenticity of the license verification.
 * @param cryptogram [in] A 32-byte array containing the cryptogram used for final verification.
 * @param cryptogramSize [in] Size of the 'cryptogram' buffer.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_FinalizeCheckLicenseWithCard(unsigned char *cryptogram, int cryptogramSize);

/**
 * Retrieves the license expiry date.
 * @param expiryDate [out] The license expiry date.
 * @param expiryDateSize [in,out] Size of the 'expiryDate' buffer. Receives the minimum required buffer size to hold the 'expiryDate' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'expiryDate' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetExpiryDate(char *expiryDate, int *expiryDateSize);

/**
 * Retrieves the hardware code of the external drive you run the library on. This function works on Windows only.
 * @param eHardwareCodeType [in] The requested hardware code type. Must be windowsUsb.
 * @param code [out] The device hardware code.
 * @param codeSize [in,out] Size of the 'code' buffer. Receives the minimum required buffer size to hold the 'code' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'code' (excluding terminating NULL character).
 * @param externalDrivePath [in] On Windows to activate a license on a USB device, it must contain the path to the requested USB drive (e.g 'G:\').
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetExternalDriveHardwareCode(id3FaceLicenseHardwareCodeType eHardwareCodeType, char *code, int *codeSize, const char *externalDrivePath);

/**
 * Retrieves the hardware code of the device using the requested hardware code type.
 * This function retrieves the hardware code of the device you run the library on.
 * The optional parameter is required:
 * - On Android, it must be the JNIEnv* pointer cast as a void*.
 * - In other cases, it must be set to NULL.
 * @param eHardwareCodeType [in] The requested hardware code type.
 * @param code [out] The device hardware code.
 * @param codeSize [in,out] Size of the 'code' buffer. Receives the minimum required buffer size to hold the 'code' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'code' (excluding terminating NULL character).
 * @param optionalParameter [in] Only used on Android for JNIEnv* pointer. Ignored for other devices (NULL).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetHostHardwareCode(id3FaceLicenseHardwareCodeType eHardwareCodeType, char *code, int *codeSize, void *optionalParameter);

/**
 * Retrieves the hardware code of the previously loaded license.
 * @param licenseFileHardwareCode [out] The license hardware code.
 * @param licenseFileHardwareCodeSize [in,out] Size of the 'licenseFileHardwareCode' buffer. Receives the minimum required buffer size to hold the 'licenseFileHardwareCode' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'licenseFileHardwareCode' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetLicenseFileHardwareCode(char *licenseFileHardwareCode, int *licenseFileHardwareCodeSize);

/**
 * Retrieves the serial number of the previously loaded license.
 * @param licenseFileSerial [out] The license serial number.
 * @param licenseFileSerialSize [in,out] Size of the 'licenseFileSerial' buffer. Receives the minimum required buffer size to hold the 'licenseFileSerial' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'licenseFileSerial' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetLicenseFileSerial(char *licenseFileSerial, int *licenseFileSerialSize);

/**
 * Retrieves the license name.
 * @param licenseName [out] The license name.
 * @param licenseNameSize [in,out] Size of the 'licenseName' buffer. Receives the minimum required buffer size to hold the 'licenseName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'licenseName' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetLicenseName(char *licenseName, int *licenseNameSize);

/**
 * Retrieves the license owner.
 * @param licenseOwner [out] The license owner.
 * @param licenseOwnerSize [in,out] Size of the 'licenseOwner' buffer. Receives the minimum required buffer size to hold the 'licenseOwner' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'licenseOwner' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetLicenseOwner(char *licenseOwner, int *licenseOwnerSize);

/**
 * Retrieves the license path.
 * @param licensePath [out] The license path.
 * @param licensePathSize [in,out] Size of the 'licensePath' buffer. Receives the minimum required buffer size to hold the 'licensePath' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'licensePath' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetLicensePath(char *licensePath, int *licensePathSize);

/**
 * Retrieves the license type.
 * @param eFaceLicenseType [out] The license type.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetLicenseType(id3FaceLicenseType *eFaceLicenseType);

/**
 * Retrieves the number of modules in the license.
 * @param count [out] The number of modules in the license.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetModuleCount(int *count);

/**
 * Retrieves the name of the specified license module.
 * @param index [in] Index of the module, starting from 0.
 * @param name [out] The license module name.
 * @param nameSize [in,out] Size of the 'name' buffer. Receives the minimum required buffer size to hold the 'name' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'name' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetModuleName(int index, char *name, int *nameSize);

/**
 * Retrieves the value associated to the specified license module.
 * @param name [in] Name of the requested module.
 * @param value [out] The value associated to the licence module.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetModuleValue(const char *name, int *value);

/**
 * Retrieves the license product reference.
 * @param productReference [out] The product reference.
 * @param productReferenceSize [in,out] Size of the 'productReference' buffer. Receives the minimum required buffer size to hold the 'productReference' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'productReference' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetProductReference(char *productReference, int *productReferenceSize);

/**
 * Retrieves the number of remaining days.
 * @param remainingDays [out] The number of remaining days.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_GetRemainingDays(int *remainingDays);

/**
 * Reactivates a license file using the host hardware code and the product reference.
 * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
 * @param productReference [in] The product reference.
 * @param path [in] Path to the license file.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_Reactivate(const char *hardwareCode, const char *productReference, const char *path);

/**
 * Reactivates a license in a data buffer using the host hardware code and the product reference.
 * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
 * @param productReference [in] The product reference.
 * @param licenseFileBuffer [out] The new license data buffer.
 * @param licenseFileBufferSize [in,out] Size of the 'licenseFileBuffer' buffer. Receives the number of elements in the 'licenseFileBuffer' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceLicense_ReactivateBuffer(const char *hardwareCode, const char *productReference, unsigned char *licenseFileBuffer, int *licenseFileBufferSize);

#ifdef __cplusplus
}
#endif

#endif
