//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FacePortrait_h
#define _id3FacePortrait_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceBackgroundUniformity.h>
#include <id3Face/id3FaceExpression.h>
#include <id3Face/id3FaceEyeGaze.h>
#include <id3Face/id3FaceTemplate.h>
#include <id3Face/id3FaceGeometricAttributes.h>
#include <id3Face/id3FaceImage.h>
#include <id3Face/id3FacePoint.h>
#include <id3Face/id3FacePortraitInstruction.h>
#include <id3Face/id3FacePointList.h>
#include <id3Face/id3FacePadStatus.h>
#include <id3Face/id3FacePhotographicAttributes.h>
#include <id3Face/id3FacePose.h>
#include <id3Face/id3FacePortraitQualityCheckpoints.h>
#include <id3Face/id3FacePortraitStatus.h>
#include <id3Face/id3FaceSubjectPosition.h>
#include <id3Face/id3TrackedFace.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a portrait.
 */

/**
 * Handle to a Portrait object of the Face module.
 */
typedef struct id3FacePortrait *ID3_FACE_PORTRAIT;

/**
 * @brief Initializes a Portrait object.
 * @param phPortrait [in] Handle to the Portrait object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_Initialize(ID3_FACE_PORTRAIT *phPortrait);

/**
 * @brief Releases the memory allocated for a Portrait.
 * @param phPortrait [in] Handle to the Portrait object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_Dispose(ID3_FACE_PORTRAIT *phPortrait);

/**
 * @brief Checks if the handle object is valid.
 * @param hPortrait [in] Handle to the Portrait object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_CheckHandle(ID3_FACE_PORTRAIT hPortrait);

/**
 * @brief Copies an instance of Portrait object to another.
 * @param hPortraitSrc [in] Handle to the source Portrait object.
 * @param hPortraitDst [in] Handle to the destination Portrait object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_CopyTo(ID3_FACE_PORTRAIT hPortraitSrc, ID3_FACE_PORTRAIT hPortraitDst);

/**
 * Gets the estimated age of the subject. -1 if not computed.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param age [out] Estimated age of the subject. -1 if not computed.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetAge(ID3_FACE_PORTRAIT hPortrait, int *age);

/**
 * Gets the result of the background uniformity computation.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param sBackgroundUniformity [out] Result of the background uniformity computation.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetBackgroundUniformity(ID3_FACE_PORTRAIT hPortrait, id3FaceBackgroundUniformity *sBackgroundUniformity);

/**
 * Gets the estimated expression of the subject (angriness, disgust, fear, happiness, sadness, surprise, or neutral). Unknown if not computed.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param eExpression [out] Estimated expression of the subject (angriness, disgust, fear, happiness, sadness, surprise, or neutral). Unknown if not computed.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetExpression(ID3_FACE_PORTRAIT hPortrait, id3FaceExpression *eExpression);

/**
 * Gets the estimated gaze.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param sEyeGaze [out] Estimated gaze.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetEyeGaze(ID3_FACE_PORTRAIT hPortrait, id3FaceEyeGaze *sEyeGaze);

/**
 * Gets the unique face ID.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param faceId [out] Unique face ID.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetFaceId(ID3_FACE_PORTRAIT hPortrait, int *faceId);

/**
 * Gets the value indicating whether a face mask is detected. The range is 0-100, 100 meaning a mask is detected.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param faceMask [out] Value indicating whether a face mask is detected. The range is 0-100, 100 meaning a mask is detected.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetFaceMask(ID3_FACE_PORTRAIT hPortrait, int *faceMask);

/**
 * Gets the face template.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param hFaceTemplate [out] Face template.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetFaceTemplate(ID3_FACE_PORTRAIT hPortrait, ID3_FACE_TEMPLATE hFaceTemplate);

/**
 * Gets the value indicating whether the subject is a male. -1 if not computed. The range is 0 (female) -100 (male).
 * @param hPortrait [in] Handle to the Portrait object.
 * @param genderMale [out] Value indicating whether the subject is a male. -1 if not computed. The range is 0 (female) -100 (male).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetGenderMale(ID3_FACE_PORTRAIT hPortrait, int *genderMale);

/**
 * Gets the geometric attributes.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param sGeometricAttributes [out] Geometric attributes.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetGeometricAttributes(ID3_FACE_PORTRAIT hPortrait, id3FaceGeometricAttributes *sGeometricAttributes);

/**
 * Gets the value indicating whether the subject wears glasses. -1 if not computed. The range is 0-100, 100 meaning glasses are detected.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param glasses [out] Value indicating whether the subject wears glasses. -1 if not computed. The range is 0-100, 100 meaning glasses are detected.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetGlasses(ID3_FACE_PORTRAIT hPortrait, int *glasses);

/**
 * Gets the value indicating whether the subject wears a hat. -1 if not computed. The range is 0-100, 100 meaning a hat is detected.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param hat [out] Value indicating whether the subject wears a hat. -1 if not computed. The range is 0-100, 100 meaning a hat is detected.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetHat(ID3_FACE_PORTRAIT hPortrait, int *hat);

/**
 * Gets the source image.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param hImage [out] Source image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetImage(ID3_FACE_PORTRAIT hPortrait, ID3_FACE_IMAGE hImage);

/**
 * Gets the center of the portrait in the source image.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param sImageCenter [out] Center of the portrait in the source image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetImageCenter(ID3_FACE_PORTRAIT hPortrait, id3FacePoint *sImageCenter);

/**
 * Gets the scale to be applied to the source image to display the portrait.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param imageScale [out] Scale to be applied to the source image to display the portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetImageScale(ID3_FACE_PORTRAIT hPortrait, float *imageScale);

/**
 * Gets the translation to be applied to the source image to display the portrait.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param sImageTranslation [out] Translation to be applied to the source image to display the portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetImageTranslation(ID3_FACE_PORTRAIT hPortrait, id3FacePoint *sImageTranslation);

/**
 * Gets the instruction to be given to the user for portrait capture.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param eInstruction [out] Instruction to be given to the user for portrait capture.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetInstruction(ID3_FACE_PORTRAIT hPortrait, id3FacePortraitInstruction *eInstruction);

/**
 * Gets the 68-points landmarks.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param hLandmarks [out] 68-points landmarks.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetLandmarks(ID3_FACE_PORTRAIT hPortrait, ID3_FACE_POINT_LIST hLandmarks);

/**
 * Gets the value estimating the opening of the left eye. -1 if not computed. The range is 0-100, 100 meaning the eye is fully opened.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param leftEyeOpening [out] Value estimating the opening of the left eye. -1 if not computed. The range is 0-100, 100 meaning the eye is fully opened.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetLeftEyeOpening(ID3_FACE_PORTRAIT hPortrait, int *leftEyeOpening);

/**
 * Gets the value estimating the visibility of the left eye. -1 if not computed. The range is 0-100, 100 meaning the left eye is fully visible.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param leftEyeVisibility [out] Value estimating the visibility of the left eye. -1 if not computed. The range is 0-100, 100 meaning the left eye is fully visible.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetLeftEyeVisibility(ID3_FACE_PORTRAIT hPortrait, int *leftEyeVisibility);

/**
 * Gets the value estimating if the user is looking straight towards the camera. -1 if not computed. The range is 0-100, 100 meaning the look is straight.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param lookStraightScore [out] Value estimating if the user is looking straight towards the camera. -1 if not computed. The range is 0-100, 100 meaning the look is straight.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetLookStraightScore(ID3_FACE_PORTRAIT hPortrait, int *lookStraightScore);

/**
 * Gets the value indicating whether makeup is detected. -1 if not computed. The range is 0-100, 100 meaning makeup is detected.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param makeup [out] Value indicating whether makeup is detected. -1 if not computed. The range is 0-100, 100 meaning makeup is detected.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetMakeup(ID3_FACE_PORTRAIT hPortrait, int *makeup);

/**
 * Gets the value indicating estimating the opening of the mouth. -1 if not computed. The range is 0-100, 100 meaning the mouth is fully open.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param mouthOpening [out] Value indicating estimating the opening of the mouth. -1 if not computed. The range is 0-100, 100 meaning the mouth is fully open.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetMouthOpening(ID3_FACE_PORTRAIT hPortrait, int *mouthOpening);

/**
 * Gets the value estimating the visibility of the mouth. -1 if not computed. The range is 0-100, 100 meaning the mouth is fully visible.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param mouthVisibility [out] Value estimating the visibility of the mouth. -1 if not computed. The range is 0-100, 100 meaning the mouth is fully visible.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetMouthVisibility(ID3_FACE_PORTRAIT hPortrait, int *mouthVisibility);

/**
 * Gets the value estimating the visibility of the nose. -1 if not computed. The range is 0-100, 100 meaning the nose is fully visible.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param noseVisibility [out] Value estimating the visibility of the nose. -1 if not computed. The range is 0-100, 100 meaning the nose is fully visible.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetNoseVisibility(ID3_FACE_PORTRAIT hPortrait, int *noseVisibility);

/**
 * Gets the score of the presentation attack detection.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param padScore [out] Score of the presentation attack detection.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetPadScore(ID3_FACE_PORTRAIT hPortrait, int *padScore);

/**
 * Gets the status of the presentation attack detection.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param ePadStatus [out] Status of the presentation attack detection.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetPadStatus(ID3_FACE_PORTRAIT hPortrait, id3FacePadStatus *ePadStatus);

/**
 * Gets the photographic attributes.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param sPhotographicAttributes [out] Photographic attributes.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetPhotographicAttributes(ID3_FACE_PORTRAIT hPortrait, id3FacePhotographicAttributes *sPhotographicAttributes);

/**
 * Gets the face pose.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param sPose [out] Face pose.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetPose(ID3_FACE_PORTRAIT hPortrait, id3FacePose *sPose);

/**
 * Gets the quality check points of the portrait.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param sQualityCheckpoints [out] Quality check points of the portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetQualityCheckpoints(ID3_FACE_PORTRAIT hPortrait, id3FacePortraitQualityCheckpoints *sQualityCheckpoints);

/**
 * Gets the unified quality score, from 0 to 100. -1 if not computed.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param qualityScore [out] Unified quality score, from 0 to 100. -1 if not computed.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetQualityScore(ID3_FACE_PORTRAIT hPortrait, int *qualityScore);

/**
 * Gets the value estimating the opening of the left eye. -1 if not computed. The range is 0-100, 100 meaning the eye is fully opened.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param rightEyeOpening [out] Value estimating the opening of the left eye. -1 if not computed. The range is 0-100, 100 meaning the eye is fully opened.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetRightEyeOpening(ID3_FACE_PORTRAIT hPortrait, int *rightEyeOpening);

/**
 * Gets the value estimating the visibility of the right eye. -1 if not computed. The range is 0-100, 100 meaning the right eye is fully visible.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param rightEyeVisibility [out] Value estimating the visibility of the right eye. -1 if not computed. The range is 0-100, 100 meaning the right eye is fully visible.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetRightEyeVisibility(ID3_FACE_PORTRAIT hPortrait, int *rightEyeVisibility);

/**
 * Gets the value indicating whether the subject is smiling. -1 if not computed. The range is 0-100, 100 meaning the person is smiling.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param smile [out] Value indicating whether the subject is smiling. -1 if not computed. The range is 0-100, 100 meaning the person is smiling.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetSmile(ID3_FACE_PORTRAIT hPortrait, int *smile);

/**
 * Gets the status of the portrait.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param eStatus [out] Status of the portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetStatus(ID3_FACE_PORTRAIT hPortrait, id3FacePortraitStatus *eStatus);

/**
 * Gets the position of the subject in the image.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param eSubjectPosition [out] Position of the subject in the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetSubjectPosition(ID3_FACE_PORTRAIT hPortrait, id3FaceSubjectPosition *eSubjectPosition);

/**
 * Gets the tracked face.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param hTrackedFace [out] Tracked face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_GetTrackedFace(ID3_FACE_PORTRAIT hPortrait, ID3_TRACKED_FACE hTrackedFace);

/**
 * Clears all attributes.
 * @param hPortrait [in] Handle to the Portrait object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_Clear(ID3_FACE_PORTRAIT hPortrait);

/**
 * Imports the portrait object from a buffer.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param data [in] Buffer to import the portrait object from.
 * @param dataSize [in] Size of the 'data' buffer.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_FromBuffer(ID3_FACE_PORTRAIT hPortrait, unsigned char *data, int dataSize);

/**
 * Exports the portrait to a buffer.
 * @param hPortrait [in] Handle to the Portrait object.
 * @param data [out] The buffer to which the portrait object is exported.
 * @param dataSize [in,out] Size of the 'data' buffer. Receives the number of elements in the 'data' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortrait_ToBuffer(ID3_FACE_PORTRAIT hPortrait, unsigned char *data, int *dataSize);

#ifdef __cplusplus
}
#endif

#endif
