//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FaceTemplate_h
#define _id3FaceTemplate_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceTemplateFormat.h>
#include <id3Face/id3FaceTemplate.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a face template.
 */

/**
 * Handle to a FaceTemplate object of the Face module.
 */
typedef struct id3FaceTemplate *ID3_FACE_TEMPLATE;

/**
 * @brief Initializes a FaceTemplate object.
 * @param phFaceTemplate [in] Handle to the FaceTemplate object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_Initialize(ID3_FACE_TEMPLATE *phFaceTemplate);

/**
 * @brief Releases the memory allocated for a FaceTemplate.
 * @param phFaceTemplate [in] Handle to the FaceTemplate object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_Dispose(ID3_FACE_TEMPLATE *phFaceTemplate);

/**
 * @brief Checks if the handle object is valid.
 * @param hFaceTemplate [in] Handle to the FaceTemplate object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_CheckHandle(ID3_FACE_TEMPLATE hFaceTemplate);

/**
 * @brief Copies an instance of FaceTemplate object to another.
 * @param hFaceTemplateSrc [in] Handle to the source FaceTemplate object.
 * @param hFaceTemplateDst [in] Handle to the destination FaceTemplate object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_CopyTo(ID3_FACE_TEMPLATE hFaceTemplateSrc, ID3_FACE_TEMPLATE hFaceTemplateDst);

/**
 * Gets the format of the face template.
 * @param hFaceTemplate [in] Handle to the FaceTemplate object.
 * @param eFormat [out] Format of the face template.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_GetFormat(ID3_FACE_TEMPLATE hFaceTemplate, id3FaceTemplateFormat *eFormat);

/**
 * Gets the quality of the face template.
 * @param hFaceTemplate [in] Handle to the FaceTemplate object.
 * @param quality [out] Quality of the face template.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_GetQuality(ID3_FACE_TEMPLATE hFaceTemplate, int *quality);

/**
 * Imports the face template object from a buffer.
 * @param hFaceTemplate [in] Handle to the FaceTemplate object.
 * @param data [in] Buffer to import the face template object from.
 * @param dataSize [in] Size of the 'data' buffer.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_FromBuffer(ID3_FACE_TEMPLATE hFaceTemplate, unsigned char *data, int dataSize);

/**
 * Imports the face template object from a file.
 * @param hFaceTemplate [in] Handle to the FaceTemplate object.
 * @param path [in] Path to the file to import the face template object from.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_FromFile(ID3_FACE_TEMPLATE hFaceTemplate, const char *path);

/**
 * Exports the face template object to a Biometric Data Template (BDT) buffer.
 * This buffer can only be used with id3 Face Match on Card specific implementations to verify a face template.
 * @param hFaceTemplate [in] Handle to the FaceTemplate object.
 * @param data [out] A buffer that receives the biometric data template.
 * @param dataSize [in,out] Size of the 'data' buffer. Receives the number of elements in the 'data' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_ToBdt(ID3_FACE_TEMPLATE hFaceTemplate, unsigned char *data, int *dataSize);

/**
 * Exports the face template as a Biometric Information Template (BIT) for enrolment on a smart card equipped with id3 Match-on-Card technology.
 * Note: The threshold value is required and should be set according to the used Face Encoder and desired security level.
 * Important: The reference data qualifier (RDQ) should be defined in accordance with the smart card application specifications.
 * @param hFaceTemplate [in] Handle to the FaceTemplate object.
 * @param threshold [in] The decision threshold, from 0 to 653535, to be applied during the face comparison on the smart card. See FaceMatcherThreshold for a list of typical values.
 * @param referenceDataQualifier [in] Reference data qualifier.
 * @param data [out] A buffer that receives the biometric information template.
 * @param dataSize [in,out] Size of the 'data' buffer. Receives the number of elements in the 'data' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_ToBit(ID3_FACE_TEMPLATE hFaceTemplate, int threshold, int referenceDataQualifier, unsigned char *data, int *dataSize);

/**
 * Exports the face template object to a buffer.
 * @param hFaceTemplate [in] Handle to the FaceTemplate object.
 * @param data [out] The buffer that receives the face template.
 * @param dataSize [in,out] Size of the 'data' buffer. Receives the number of elements in the 'data' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_ToBuffer(ID3_FACE_TEMPLATE hFaceTemplate, unsigned char *data, int *dataSize);

/**
 * Exports the face template object to a file.
 * @param hFaceTemplate [in] Handle to the FaceTemplate object.
 * @param path [in] Path to the file to export the face template object to.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_ToFile(ID3_FACE_TEMPLATE hFaceTemplate, const char *path);

/**
 * Updates the face template by combining it with the specified face template.
 * @param hFaceTemplate [in] Handle to the FaceTemplate object.
 * @param hNewFaceTemplate [in] New face template to combine.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplate_Update(ID3_FACE_TEMPLATE hFaceTemplate, ID3_FACE_TEMPLATE hNewFaceTemplate);

#ifdef __cplusplus
}
#endif

#endif
