//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FaceTracker_h
#define _id3FaceTracker_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceModel.h>
#include <id3Face/id3FaceImage.h>
#include <id3Face/id3TrackedFaceList.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Tracks faces on consecutive images and automatically creates and updates associated face templates.
 */

/**
 * Handle to a FaceTracker object of the Face module.
 */
typedef struct id3FaceTracker *ID3_FACE_TRACKER;

/**
 * @brief Initializes a FaceTracker object.
 * @param phFaceTracker [in] Handle to the FaceTracker object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_Initialize(ID3_FACE_TRACKER *phFaceTracker);

/**
 * @brief Releases the memory allocated for a FaceTracker.
 * @param phFaceTracker [in] Handle to the FaceTracker object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_Dispose(ID3_FACE_TRACKER *phFaceTracker);

/**
 * @brief Checks if the handle object is valid.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_CheckHandle(ID3_FACE_TRACKER hFaceTracker);

/**
 * Gets the confidence threshold, in the range [0;100].
 * Hint: Default value is 50.
 * Note: Setting a high threshold reduces false detection but can increase the number of undetected faces.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param confidenceThreshold [out] Confidence threshold, in the range [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_GetConfidenceThreshold(ID3_FACE_TRACKER hFaceTracker, int *confidenceThreshold);

/**
 * Sets the confidence threshold, in the range [0;100].
 * Hint: Default value is 50.
 * Note: Setting a high threshold reduces false detection but can increase the number of undetected faces.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param confidenceThreshold [in] Confidence threshold, in the range [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_SetConfidenceThreshold(ID3_FACE_TRACKER hFaceTracker, int confidenceThreshold);

/**
 * Gets the model used to detect and track faces.
 * Hint: Default value is FaceDetector4B.
 * Note: Some better accuracy/speed balances can be found by choosing another model.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param eDetectionModel [out] Model used to detect and track faces.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_GetDetectionModel(ID3_FACE_TRACKER hFaceTracker, id3FaceModel *eDetectionModel);

/**
 * Sets the model used to detect and track faces.
 * Hint: Default value is FaceDetector4B.
 * Note: Some better accuracy/speed balances can be found by choosing another model.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param eDetectionModel [in] Model used to detect and track faces.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_SetDetectionModel(ID3_FACE_TRACKER hFaceTracker, id3FaceModel eDetectionModel);

/**
 * Gets the model used to create features and assess consistancy among views of a given face.
 * Hint: Default value is FaceEncoder9B. Some better accuracy/speed balances can be found by choosing another model.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param eEncodingModel [out] Model used to create features and assess consistancy among views of a given face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_GetEncodingModel(ID3_FACE_TRACKER hFaceTracker, id3FaceModel *eEncodingModel);

/**
 * Sets the model used to create features and assess consistancy among views of a given face.
 * Hint: Default value is FaceEncoder9B. Some better accuracy/speed balances can be found by choosing another model.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param eEncodingModel [in] Model used to create features and assess consistancy among views of a given face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_SetEncodingModel(ID3_FACE_TRACKER hFaceTracker, id3FaceModel eEncodingModel);

/**
 * Gets the minimum match score to reach to preserve the ID of a tracked face between frame 't-1' and frame 't'.
 * Hint: Default value is 3000 which corresponds to a False Match Rate of 1/1000.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param matchThreshold [out] Minimum match score to reach to preserve the ID of a tracked face between frame 't-1' and frame 't'.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_GetMatchThreshold(ID3_FACE_TRACKER hFaceTracker, int *matchThreshold);

/**
 * Sets the minimum match score to reach to preserve the ID of a tracked face between frame 't-1' and frame 't'.
 * Hint: Default value is 3000 which corresponds to a False Match Rate of 1/1000.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param matchThreshold [in] Minimum match score to reach to preserve the ID of a tracked face between frame 't-1' and frame 't'.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_SetMatchThreshold(ID3_FACE_TRACKER hFaceTracker, int matchThreshold);

/**
 * Gets the maximum number of consecutive non-detections to reach before deleting a tracked face.
 * Hint: Default value is 30 which corresponds to 2s at a frame rate of 15 FPS. One must adapt this value to its needs in terms of tracker identity memory (in seconds) and measured frame rate on target platform.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param maximumTrackedFaceAge [out] Maximum number of consecutive non-detections to reach before deleting a tracked face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_GetMaximumTrackedFaceAge(ID3_FACE_TRACKER hFaceTracker, int *maximumTrackedFaceAge);

/**
 * Sets the maximum number of consecutive non-detections to reach before deleting a tracked face.
 * Hint: Default value is 30 which corresponds to 2s at a frame rate of 15 FPS. One must adapt this value to its needs in terms of tracker identity memory (in seconds) and measured frame rate on target platform.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param maximumTrackedFaceAge [in] Maximum number of consecutive non-detections to reach before deleting a tracked face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_SetMaximumTrackedFaceAge(ID3_FACE_TRACKER hFaceTracker, int maximumTrackedFaceAge);

/**
 * Gets the minimum number of consecutive detections to reach before creating a tracked face.
 * Hint: Default value is 1 for FaceDetector4B since the false detection rate is low enough. If using a less accurate detector (such as FaceDetector3C) one might consider increasing a bit this value to avoid false tracks.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param minimumTrackedFaceAge [out] Minimum number of consecutive detections to reach before creating a tracked face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_GetMinimumTrackedFaceAge(ID3_FACE_TRACKER hFaceTracker, int *minimumTrackedFaceAge);

/**
 * Sets the minimum number of consecutive detections to reach before creating a tracked face.
 * Hint: Default value is 1 for FaceDetector4B since the false detection rate is low enough. If using a less accurate detector (such as FaceDetector3C) one might consider increasing a bit this value to avoid false tracks.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param minimumTrackedFaceAge [in] Minimum number of consecutive detections to reach before creating a tracked face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_SetMinimumTrackedFaceAge(ID3_FACE_TRACKER hFaceTracker, int minimumTrackedFaceAge);

/**
 * Gets the non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range is [0;100].
 * Hint: Default value is 40. Setting a high threshold allows to detect more overlapping faces which can be useful in a multi-face scenario. On the contrary, in a portrait scenario, a low NMS IOU threshold should be preferred.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param nmsIouThreshold [out] Non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range is [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_GetNmsIouThreshold(ID3_FACE_TRACKER hFaceTracker, int *nmsIouThreshold);

/**
 * Sets the non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range is [0;100].
 * Hint: Default value is 40. Setting a high threshold allows to detect more overlapping faces which can be useful in a multi-face scenario. On the contrary, in a portrait scenario, a low NMS IOU threshold should be preferred.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param nmsIouThreshold [in] Non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range is [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_SetNmsIouThreshold(ID3_FACE_TRACKER hFaceTracker, int nmsIouThreshold);

/**
 * Gets the number of threads to be used for face detection and tracking.
 * Hint: Default value is 1. Allocating more than one thread can increase the speed of the process.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param threadCount [out] Number of threads to be used for face detection and tracking.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_GetThreadCount(ID3_FACE_TRACKER hFaceTracker, int *threadCount);

/**
 * Sets the number of threads to be used for face detection and tracking.
 * Hint: Default value is 1. Allocating more than one thread can increase the speed of the process.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param threadCount [in] Number of threads to be used for face detection and tracking.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_SetThreadCount(ID3_FACE_TRACKER hFaceTracker, int threadCount);

/**
 * Detects faces in an image and update their info in a TrackedFaceList object.
 * The algorithm searches for faces in the range [16px;512px]. If the image is too large to fit this range, one must resize it before the tracking process.
 * In a realtime process, one must use this function to keep the face IDs stable in time.
 * Important: Loading a ``FaceDetector`` is required to use this function.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param hImage [in] Source image to process.
 * @param hTrackedFaceList [in] List of tracked faces.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_DetectFaces(ID3_FACE_TRACKER hFaceTracker, ID3_FACE_IMAGE hImage, ID3_TRACKED_FACE_LIST hTrackedFaceList);

/**
 * Tracks faces in an image and update their info in a TrackedFaceList object.
 * The algorithm searches for faces in the range [16px;512px]. If the image is too large to fit this range, one must resize it before the tracking process.
 * In a realtime process, one must use this function to keep the face IDs stable in time.
 * Important: Loading a ``FaceDetector`` and a ``FaceEncoder`` model is required to use this function.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param hImage [in] Source image to process.
 * @param hTrackedFaceList [in] List of tracked faces.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_TrackFaces(ID3_FACE_TRACKER hFaceTracker, ID3_FACE_IMAGE hImage, ID3_TRACKED_FACE_LIST hTrackedFaceList);

/**
 * Gets the face tracker module ready to work on a specific image size by initializing all its internal memory layout.
 * This function can be called after setting all the parameters of the face detector module if the image size is fixed, for example using frames from a video stream.
 * Calling this function is not mandatory as the internal memory layout is automatically initialized in the first call to detectFaces() or trackFaces() if missing.
 * Important: Loading a ``FaceDetector`` model is required to use this function.
 * @param hFaceTracker [in] Handle to the FaceTracker object.
 * @param imageWidth [in] Width of the expected image size.
 * @param imageHeight [in] Height of expected image size.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTracker_WarmUp(ID3_FACE_TRACKER hFaceTracker, int imageWidth, int imageHeight);

#ifdef __cplusplus
}
#endif

#endif
