//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3TrackedFace_h
#define _id3TrackedFace_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceRectangle.h>
#include <id3Face/id3FaceTemplate.h>
#include <id3Face/id3FacePointList.h>
#include <id3Face/id3FaceTrackingStatus.h>
#include <id3Face/id3DetectedFace.h>
#include <id3Face/id3FaceImage.h>
#include <id3Face/id3FacePoint.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a tracked face.
 */

/**
 * Handle to a TrackedFace object of the Face module.
 */
typedef struct id3TrackedFace *ID3_TRACKED_FACE;

/**
 * @brief Initializes a TrackedFace object.
 * @param phTrackedFace [in] Handle to the TrackedFace object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_Initialize(ID3_TRACKED_FACE *phTrackedFace);

/**
 * @brief Releases the memory allocated for a TrackedFace.
 * @param phTrackedFace [in] Handle to the TrackedFace object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_Dispose(ID3_TRACKED_FACE *phTrackedFace);

/**
 * @brief Checks if the handle object is valid.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_CheckHandle(ID3_TRACKED_FACE hTrackedFace);

/**
 * @brief Copies an instance of TrackedFace object to another.
 * @param hTrackedFaceSrc [in] Handle to the source TrackedFace object.
 * @param hTrackedFaceDst [in] Handle to the destination TrackedFace object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_CopyTo(ID3_TRACKED_FACE hTrackedFaceSrc, ID3_TRACKED_FACE hTrackedFaceDst);

/**
 * Gets the bounds of the detected face.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param sBounds [out] Bounds of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetBounds(ID3_TRACKED_FACE hTrackedFace, id3FaceRectangle *sBounds);

/**
 * Sets the bounds of the detected face.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param sBounds [in] Bounds of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_SetBounds(ID3_TRACKED_FACE hTrackedFace, const id3FaceRectangle *sBounds);

/**
 * Gets the confidence score of the detected face.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param detectionScore [out] Confidence score of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetDetectionScore(ID3_TRACKED_FACE hTrackedFace, int *detectionScore);

/**
 * Gets the computed template the tracked face. It aims at tracking an identity over time and is consolidated.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param hFaceTemplate [out] Computed template the tracked face. It aims at tracking an identity over time and is consolidated.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetFaceTemplate(ID3_TRACKED_FACE hTrackedFace, ID3_FACE_TEMPLATE hFaceTemplate);

/**
 * Gets the ID of the detected face.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param id [out] ID of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetId(ID3_TRACKED_FACE hTrackedFace, int *id);

/**
 * Sets the ID of the detected face.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param id [in] ID of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_SetId(ID3_TRACKED_FACE hTrackedFace, int id);

/**
 * Gets the landmarks (eyes, nose and mouth corners) of the detected face.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param hLandmarks [out] Landmarks (eyes, nose and mouth corners) of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetLandmarks(ID3_TRACKED_FACE hTrackedFace, ID3_FACE_POINT_LIST hLandmarks);

/**
 * Sets the landmarks (eyes, nose and mouth corners) of the detected face.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param hLandmarks [in] Landmarks (eyes, nose and mouth corners) of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_SetLandmarks(ID3_TRACKED_FACE hTrackedFace, ID3_FACE_POINT_LIST hLandmarks);

/**
 * Gets the predicted bounds of the tracked face. Those bounds are computed using a Kalman filter which has the effect of making them smooth and robust to false non-detections.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param sPredictedBounds [out] Predicted bounds of the tracked face. Those bounds are computed using a Kalman filter which has the effect of making them smooth and robust to false non-detections.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetPredictedBounds(ID3_TRACKED_FACE hTrackedFace, id3FaceRectangle *sPredictedBounds);

/**
 * Gets the status of the tracked face. See FaceTrackingStatus for more details.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param eTrackingStatus [out] Status of the tracked face. See FaceTrackingStatus for more details.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetTrackingStatus(ID3_TRACKED_FACE hTrackedFace, id3FaceTrackingStatus *eTrackingStatus);

/**
 * Creates a detected face.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param sBounds [in] Bounds of the detected face.
 * @param detectionScore [in] Confidence score of the detected face.
 * @param id [in] ID of the detected face.
 * @param hLandmarks [in] Landmarks (eyes, nose and mouth corners) of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_Create(ID3_TRACKED_FACE hTrackedFace, const id3FaceRectangle *sBounds, int detectionScore, int id, ID3_FACE_POINT_LIST hLandmarks);

/**
 * Imports the face object from a buffer.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param data [in] Buffer to import the face object from.
 * @param dataSize [in] Size of the 'data' buffer.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_FromBuffer(ID3_TRACKED_FACE hTrackedFace, unsigned char *data, int dataSize);

/**
 * Imports the face object from a file.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param path [in] Path to the file to import the face object from.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_FromFile(ID3_TRACKED_FACE hTrackedFace, const char *path);

/**
 * Gets the tracked face as a detected face.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param hDetectedFace [out] The tracked face as a detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetAsDetectedFace(ID3_TRACKED_FACE hTrackedFace, ID3_DETECTED_FACE hDetectedFace);

/**
 * Gets the distance between the detected face and the camera when using a depth map in pixels.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param hDepthImage [in] Depth image to process.
 * @param distanceToCamera [out] The estimated distance to camera in pixels.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetDistanceToCamera(ID3_TRACKED_FACE hTrackedFace, ID3_FACE_IMAGE hDepthImage, int *distanceToCamera);

/**
 * Gets the bounding box around the detected face with specified aspect ratio and specified margin.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param aspectRatio [in] Aspect ratio of the bounding box. Default recommended value is 1.33 (4/3).
 * @param margin [in] Relative margin around the detected face. Recommended value is 0.5.
 * @param sPortraitBounds [out] The portrait bounds.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetExpandedBounds(ID3_TRACKED_FACE hTrackedFace, float aspectRatio, float margin, id3FaceRectangle *sPortraitBounds);

/**
 * Gets the distance between the eyes (IOD) of the detected face in pixels.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param iod [out] The computed interocular distance (IOD) in pixels.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetInterocularDistance(ID3_TRACKED_FACE hTrackedFace, int *iod);

/**
 * Gets the bounding box of the detected face for ICAO portrait cropping.
 * This method shall be used to ensure compliance with the ICAO standard.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param eyeImageWidthRatio [in] Ratio between eye distance and image width. Must be in the range ]0;1[. Default recommended value is 0.25.
 * @param eyeImageHeightRatio [in] Ratio between eye distance to top and image height. Must be in the range ]0;1[. Default recommended value is 0.45.
 * @param imageRatio [in] Ratio between image height and image width. Default recommended value is 1.33 (4/3).
 * @param sPortraitBounds [out] The portrait bounds of the detected face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_GetPortraitBounds(ID3_TRACKED_FACE hTrackedFace, float eyeImageWidthRatio, float eyeImageHeightRatio, float imageRatio, id3FaceRectangle *sPortraitBounds);

/**
 * Rescales the detected face object bounding box and landmarks. This function can be useful if the image was downscaled to speed up detection, then you need to upscale the detected face to fit the source image size.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param scale [in] The multiplicative rescaling factor to apply to the face object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_Rescale(ID3_TRACKED_FACE hTrackedFace, float scale);

/**
 * Rotates the face object by a given angle in degrees from a given center.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param angle [in] Angle of the rotation to apply to the face object.
 * @param sCenter [in] Center of the rotation to apply to the face object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_Rotate(ID3_TRACKED_FACE hTrackedFace, int angle, const id3FacePoint *sCenter);

/**
 * Exports the face object to a buffer.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param data [out] The buffer to which the face object is exported.
 * @param dataSize [in,out] Size of the 'data' buffer. Receives the number of elements in the 'data' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_ToBuffer(ID3_TRACKED_FACE hTrackedFace, unsigned char *data, int *dataSize);

/**
 * Saves the face object to a file.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param path [in] Path to the file to export the face object to.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_ToFile(ID3_TRACKED_FACE hTrackedFace, const char *path);

/**
 * Translates the face object.
 * @param hTrackedFace [in] Handle to the TrackedFace object.
 * @param tx [in] Translation to apply to the face object alongside the x-axis.
 * @param ty [in] Translation to apply to the face object alongside the y-axis.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFace_Translate(ID3_TRACKED_FACE hTrackedFace, int tx, int ty);

#ifdef __cplusplus
}
#endif

#endif
