//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FaceEncoder_h
#define _id3FaceEncoder_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceModel.h>
#include <id3Face/id3FaceProcessingUnit.h>
#include <id3Face/id3FaceImage.h>
#include <id3Face/id3DetectedFace.h>
#include <id3Face/id3FaceTemplate.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Extracts facial features and encodes face templates.
 */

/**
 * Handle to a FaceEncoder object of the Face module.
 */
typedef struct id3FaceEncoder *ID3_FACE_ENCODER;

/**
 * @brief Initializes a FaceEncoder object.
 * @param phFaceEncoder [in] Handle to the FaceEncoder object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_Initialize(ID3_FACE_ENCODER *phFaceEncoder);

/**
 * @brief Releases the memory allocated for a FaceEncoder.
 * @param phFaceEncoder [in] Handle to the FaceEncoder object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_Dispose(ID3_FACE_ENCODER *phFaceEncoder);

/**
 * @brief Checks if the handle object is valid.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_CheckHandle(ID3_FACE_ENCODER hFaceEncoder);

/**
 * Gets the minimal interocular distance (IOD) in pixels required to perform a face encoding.
 * Hint: Default value is 30 pixels. It is not recommended to change this value.
 * Warning: Reducing the minimal required IOD to encode smaller face images may create unreliable face templates and significantly degrade recognition performances.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @param minimalIod [out] Minimal interocular distance (IOD) in pixels required to perform a face encoding.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_GetMinimalIod(ID3_FACE_ENCODER hFaceEncoder, int *minimalIod);

/**
 * Sets the minimal interocular distance (IOD) in pixels required to perform a face encoding.
 * Hint: Default value is 30 pixels. It is not recommended to change this value.
 * Warning: Reducing the minimal required IOD to encode smaller face images may create unreliable face templates and significantly degrade recognition performances.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @param minimalIod [in] Minimal interocular distance (IOD) in pixels required to perform a face encoding.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_SetMinimalIod(ID3_FACE_ENCODER hFaceEncoder, int minimalIod);

/**
 * Gets the model used to create templates.
 * Hint: Default value is FaceEncoder9A which is the best available algorithm in this SDK in terms of accuracy. Some better accuracy/speed balances can be found by choosing another model.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @param eModel [out] Model used to create templates.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_GetModel(ID3_FACE_ENCODER hFaceEncoder, id3FaceModel *eModel);

/**
 * Sets the model used to create templates.
 * Hint: Default value is FaceEncoder9A which is the best available algorithm in this SDK in terms of accuracy. Some better accuracy/speed balances can be found by choosing another model.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @param eModel [in] Model used to create templates.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_SetModel(ID3_FACE_ENCODER hFaceEncoder, id3FaceModel eModel);

/**
 * Gets the processing unit where to run the encoding process.
 * Hint: Default value is CPU.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @param eProcessingUnit [out] Processing unit where to run the encoding process.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_GetProcessingUnit(ID3_FACE_ENCODER hFaceEncoder, id3FaceProcessingUnit *eProcessingUnit);

/**
 * Sets the processing unit where to run the encoding process.
 * Hint: Default value is CPU.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @param eProcessingUnit [in] Processing unit where to run the encoding process.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_SetProcessingUnit(ID3_FACE_ENCODER hFaceEncoder, id3FaceProcessingUnit eProcessingUnit);

/**
 * Gets the number of threads to be used for template encoding.
 * Hint: Default value is 1. Allocating more than one thread can increase the speed of the process.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @param threadCount [out] Number of threads to be used for template encoding.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_GetThreadCount(ID3_FACE_ENCODER hFaceEncoder, int *threadCount);

/**
 * Sets the number of threads to be used for template encoding.
 * Hint: Default value is 1. Allocating more than one thread can increase the speed of the process.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @param threadCount [in] Number of threads to be used for template encoding.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_SetThreadCount(ID3_FACE_ENCODER hFaceEncoder, int threadCount);

/**
 * Computes the quality of a detected face.
 * Here, quality expresses the match capability of an image: a high quality image will generate less match errors (false acceptance or false rejection) than a low quality one.
 * Warning: A minimum interocular distance (IOD) of 30 pixels for the detected face is required for this function, below this value it will output an error.
 * Important: Loading the ``FaceEncodingQualityEstimator`` model is required to use this function.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param quality [out] The estimated quality of the detected face. Range is [0:100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_ComputeQuality(ID3_FACE_ENCODER hFaceEncoder, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, int *quality);

/**
 * Extracts the unique features of a detected face and store them in a FaceTemplate.
 * Warning: A minimum interocular distance (IOD) of 30 pixels for the detected face is required for this function, below this value it will output an error.
 * Important: Loading a ``FaceEncoder`` model is required to use this function.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param hFaceTemplate [out] The generated face template.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_CreateTemplate(ID3_FACE_ENCODER hFaceEncoder, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, ID3_FACE_TEMPLATE hFaceTemplate);

/**
 * Gets the face encoder module ready to work by initializing all its internal memory layout.
 * This function can be called after setting all the parameters of the face encoder module.
 * Calling this function is not mandatory as the internal memory layout is automatically initialized in the first call to createTemplate() if missing.
 * Important: Loading a ``FaceEncoder`` is required to use this function.
 * @param hFaceEncoder [in] Handle to the FaceEncoder object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceEncoder_WarmUp(ID3_FACE_ENCODER hFaceEncoder);

#ifdef __cplusplus
}
#endif

#endif
