//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FaceTemplateDict_h
#define _id3FaceTemplateDict_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceStringList.h>
#include <id3Face/id3FaceTemplate.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a dictionary of face templates.
 */

/**
 * Handle to a FaceTemplateDict object of the Face module.
 */
typedef struct id3FaceTemplateDict *ID3_FACE_TEMPLATE_DICT;

/**
 * @brief Initializes a FaceTemplateDict object.
 * @param phFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_Initialize(ID3_FACE_TEMPLATE_DICT *phFaceTemplateDict);

/**
 * @brief Releases the memory allocated for a FaceTemplateDict.
 * @param phFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_Dispose(ID3_FACE_TEMPLATE_DICT *phFaceTemplateDict);

/**
 * @brief Checks if the handle object is valid.
 * @param hFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_CheckHandle(ID3_FACE_TEMPLATE_DICT hFaceTemplateDict);

/**
 * @brief Copies an instance of FaceTemplateDict object to another.
 * @param hFaceTemplateDictSrc [in] Handle to the source FaceTemplateDict object.
 * @param hFaceTemplateDictDst [in] Handle to the destination FaceTemplateDict object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_CopyTo(ID3_FACE_TEMPLATE_DICT hFaceTemplateDictSrc, ID3_FACE_TEMPLATE_DICT hFaceTemplateDictDst);

/**
 * Gets the gets a string list containing the keys in the dict.
 * @param hFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @param hKeys [out] Gets a string list containing the keys in the dict.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_GetKeys(ID3_FACE_TEMPLATE_DICT hFaceTemplateDict, ID3_FACE_STRING_LIST hKeys);

/**
 * Clears the FaceTemplateDict object.
 * @param hFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_Clear(ID3_FACE_TEMPLATE_DICT hFaceTemplateDict);

/**
 * Adds an item to the FaceTemplateDict object.
 * @param hFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @param key [in] Unique key of the FaceTemplate item to add.
 * @param hFaceTemplateItem [in] FaceTemplate item to add.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_Add(ID3_FACE_TEMPLATE_DICT hFaceTemplateDict, const char *key, ID3_FACE_TEMPLATE hFaceTemplateItem);

/**
 * Gets an item of the FaceTemplateDict object.
 * @param hFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @param key [in] Unique key of the FaceTemplate item to get.
 * @param hFaceTemplateItem [out] FaceTemplate item to get.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_Get(ID3_FACE_TEMPLATE_DICT hFaceTemplateDict, const char *key, ID3_FACE_TEMPLATE hFaceTemplateItem);

/**
 * Sets an item of the FaceTemplateDict object.
 * @param hFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @param key [in] Unique key of the FaceTemplate item to set.
 * @param hFaceTemplateItem [in] FaceTemplate item to set.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_Set(ID3_FACE_TEMPLATE_DICT hFaceTemplateDict, const char *key, ID3_FACE_TEMPLATE hFaceTemplateItem);

/**
 * Gets the number of elements in the FaceTemplateDict object.
 * @param hFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @param count [out] Number of elements in the FaceTemplate object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_GetCount(ID3_FACE_TEMPLATE_DICT hFaceTemplateDict, int *count);

/**
 * Determines whether the FaceTemplateDictobject contains the specified key.
 * @param hFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @param key [in] The key to locate in the FaceTemplate object.
 * @param result [out] true if the FaceTemplate object contains an element with the specified key, otherwise false.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_ContainsKey(ID3_FACE_TEMPLATE_DICT hFaceTemplateDict, const char *key, bool *result);

/**
 * Removes an element of the FaceTemplateDict object.
 * @param hFaceTemplateDict [in] Handle to the FaceTemplateDict object.
 * @param key [in] Unique key of the FaceTemplate item to remove.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceTemplateDict_Remove(ID3_FACE_TEMPLATE_DICT hFaceTemplateDict, const char *key);

#ifdef __cplusplus
}
#endif

#endif
