//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3TrackedFaceList_h
#define _id3TrackedFaceList_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3TrackedFace.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a list of tracked faces.
 */

/**
 * Handle to a TrackedFaceList object of the Face module.
 */
typedef struct id3TrackedFaceList *ID3_TRACKED_FACE_LIST;

/**
 * @brief Initializes a TrackedFaceList object.
 * @param phTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_Initialize(ID3_TRACKED_FACE_LIST *phTrackedFaceList);

/**
 * @brief Releases the memory allocated for a TrackedFaceList.
 * @param phTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_Dispose(ID3_TRACKED_FACE_LIST *phTrackedFaceList);

/**
 * @brief Checks if the handle object is valid.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_CheckHandle(ID3_TRACKED_FACE_LIST hTrackedFaceList);

/**
 * @brief Copies an instance of TrackedFaceList object to another.
 * @param hTrackedFaceListSrc [in] Handle to the source TrackedFaceList object.
 * @param hTrackedFaceListDst [in] Handle to the destination TrackedFaceList object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_CopyTo(ID3_TRACKED_FACE_LIST hTrackedFaceListSrc, ID3_TRACKED_FACE_LIST hTrackedFaceListDst);

/**
 * Clears the TrackedFaceList object.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_Clear(ID3_TRACKED_FACE_LIST hTrackedFaceList);

/**
 * Adds an item to the TrackedFaceList object.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @param hTrackedFaceItem [in] TrackedFace item to add.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_Add(ID3_TRACKED_FACE_LIST hTrackedFaceList, ID3_TRACKED_FACE hTrackedFaceItem);

/**
 * Gets an item of the TrackedFaceList object.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @param index [in] Index of the TrackedFace item to get.
 * @param hTrackedFaceItem [out] TrackedFace item to get.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_Get(ID3_TRACKED_FACE_LIST hTrackedFaceList, int index, ID3_TRACKED_FACE hTrackedFaceItem);

/**
 * Sets an item of the TrackedFaceList object.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @param index [in] Index of the TrackedFace item to set.
 * @param hTrackedFaceItem [in] TrackedFace item to set.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_Set(ID3_TRACKED_FACE_LIST hTrackedFaceList, int index, ID3_TRACKED_FACE hTrackedFaceItem);

/**
 * Gets the number of elements in the TrackedFaceList object.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @param count [out] Number of elements in the TrackedFace object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_GetCount(ID3_TRACKED_FACE_LIST hTrackedFaceList, int *count);

/**
 * Removes an element of the TrackedFaceList object.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @param index [in] Index of the TrackedFace item to remove.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_RemoveAt(ID3_TRACKED_FACE_LIST hTrackedFaceList, int index);

/**
 * TrackedFaceList object.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @param newSize [in] TrackedFace object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_Resize(ID3_TRACKED_FACE_LIST hTrackedFaceList, int newSize);

/**
 * Finds a tracked face by ID.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @param id [in] ID of the tracked face to find in the list.
 * @param hTrackedFace [out] The found tracked face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_FindTrackedFace(ID3_TRACKED_FACE_LIST hTrackedFaceList, int id, ID3_TRACKED_FACE hTrackedFace);

/**
 * Gets the list of IDs in the list.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @param ids [out] The list of IDs in the list.
 * @param idsSize [in,out] Size of the 'ids' buffer. Receives the number of elements in the 'ids' array.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_GetIdList(ID3_TRACKED_FACE_LIST hTrackedFaceList, int *ids, int *idsSize);

/**
 * Gets the largest face in the list.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @param hLargestFace [out] The largest tracked face in the list.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_GetLargestFace(ID3_TRACKED_FACE_LIST hTrackedFaceList, ID3_TRACKED_FACE hLargestFace);

/**
 * Rescales all tracked faces in the list.
 * @param hTrackedFaceList [in] Handle to the TrackedFaceList object.
 * @param scale [in] Scale factor to apply. Range is [0;+inf[.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3TrackedFaceList_RescaleAll(ID3_TRACKED_FACE_LIST hTrackedFaceList, float scale);

#ifdef __cplusplus
}
#endif

#endif
