//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FaceIndexer_h
#define _id3FaceIndexer_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceTemplateFormat.h>
#include <id3Face/id3FaceTemplate.h>
#include <id3Face/id3FaceCandidateList.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Performs a one-to-many search on a large number of face templates.
 */

/**
 * Handle to a FaceIndexer object of the Face module.
 */
typedef struct id3FaceIndexer *ID3_FACE_INDEXER;

/**
 * @brief Initializes a FaceIndexer object.
 * @param phFaceIndexer [in] Handle to the FaceIndexer object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_Initialize(ID3_FACE_INDEXER *phFaceIndexer);

/**
 * @brief Releases the memory allocated for a FaceIndexer.
 * @param phFaceIndexer [in] Handle to the FaceIndexer object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_Dispose(ID3_FACE_INDEXER *phFaceIndexer);

/**
 * @brief Checks if the handle object is valid.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_CheckHandle(ID3_FACE_INDEXER hFaceIndexer);

/**
 * Gets the face template format accepted by this face indexer.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param eFormat [out] Face template format accepted by this face indexer.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_GetFormat(ID3_FACE_INDEXER hFaceIndexer, id3FaceTemplateFormat *eFormat);

/**
 * Gets the maximum number of templates that this face indexer can hold.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param maximumTemplateCount [out] Maximum number of templates that this face indexer can hold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_GetMaximumTemplateCount(ID3_FACE_INDEXER hFaceIndexer, int *maximumTemplateCount);

/**
 * Gets the current number of indexed templates.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param templateCount [out] Current number of indexed templates.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_GetTemplateCount(ID3_FACE_INDEXER hFaceIndexer, int *templateCount);

/**
 * Adds a template to the Face Indexer.
 * Note: This function is thread safe with other AddTemplate() calls but not with SearchTemplates() calls.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param hFaceTemplate [in] Face template to add to the face indexer.
 * @param id [in] ID of the added face template.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_AddTemplate(ID3_FACE_INDEXER hFaceIndexer, ID3_FACE_TEMPLATE hFaceTemplate, const char *id);

/**
 * Creates an empty face indexer
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param maximumTemplateCount [in] Maximum number of templates that this face indexer can hold.
 * @param eFormat [in] Face template format accepted by this face indexer.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_Create(ID3_FACE_INDEXER hFaceIndexer, int maximumTemplateCount, id3FaceTemplateFormat eFormat);

/**
 * Deletes a template from the face indexer.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param id [in] ID of the face template to delete.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_DeleteTemplate(ID3_FACE_INDEXER hFaceIndexer, const char *id);

/**
 * Imports the face indexer object from a file.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param path [in] Path to the file to import the face indexer object from.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_FromFile(ID3_FACE_INDEXER hFaceIndexer, const char *path);

/**
 * Retrieves a template from the face indexer by ID.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param id [in] ID of the required face template.
 * @param hFaceTemplate [out] The face template.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_GetTemplate(ID3_FACE_INDEXER hFaceIndexer, const char *id, ID3_FACE_TEMPLATE hFaceTemplate);

/**
 * Resizes the maximum number of templates that the face indexer can hold.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param maximumCount [in] Maximum number of templates that this face indexer can hold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_Resize(ID3_FACE_INDEXER hFaceIndexer, int maximumCount);

/**
 * Performs a one-to-many search of a face template probe against a previously initialized face indexer and outputs a candidate list.
 * This function is thread safe with other SearchTemplate() calls but not with AddTemplate() calls.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param hProbe [in] The face template probe.
 * @param maxCandidates [in] Maximum number of candidates in the candidate list.
 * @param hCandidates [out] The list of candidates sorted by decreasing match score.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_SearchTemplate(ID3_FACE_INDEXER hFaceIndexer, ID3_FACE_TEMPLATE hProbe, int maxCandidates, ID3_FACE_CANDIDATE_LIST hCandidates);

/**
 * Saves the face indexer object to a file.
 * @param hFaceIndexer [in] Handle to the FaceIndexer object.
 * @param path [in] Path to the file to export the face indexer object to.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceIndexer_ToFile(ID3_FACE_INDEXER hFaceIndexer, const char *path);

#ifdef __cplusplus
}
#endif

#endif
