//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FacePad_h
#define _id3FacePad_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceImage.h>
#include <id3Face/id3DetectedFace.h>
#include <id3Face/id3FaceColorBasedPadResult.h>
#include <id3Face/id3DetectedFaceAttackSupport.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Detects presentation attacks in still images or video frames.
 */

/**
 * Handle to a FacePad object of the Face module.
 */
typedef struct id3FacePad *ID3_FACE_PAD;

/**
 * @brief Initializes a FacePad object.
 * @param phFacePad [in] Handle to the FacePad object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePad_Initialize(ID3_FACE_PAD *phFacePad);

/**
 * @brief Releases the memory allocated for a FacePad.
 * @param phFacePad [in] Handle to the FacePad object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePad_Dispose(ID3_FACE_PAD *phFacePad);

/**
 * @brief Checks if the handle object is valid.
 * @param hFacePad [in] Handle to the FacePad object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePad_CheckHandle(ID3_FACE_PAD hFacePad);

/**
 * Gets the attack support detector confidence threshold, in the range [0;100].
 * Hint: Default value is 25.
 * Note: Setting a high threshold reduces false attack support detections but can increase the number of undetected attack supports.
 * @param hFacePad [in] Handle to the FacePad object.
 * @param attackSupportDetectorConfidenceThreshold [out] Attack support detector confidence threshold, in the range [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePad_GetAttackSupportDetectorConfidenceThreshold(ID3_FACE_PAD hFacePad, int *attackSupportDetectorConfidenceThreshold);

/**
 * Sets the attack support detector confidence threshold, in the range [0;100].
 * Hint: Default value is 25.
 * Note: Setting a high threshold reduces false attack support detections but can increase the number of undetected attack supports.
 * @param hFacePad [in] Handle to the FacePad object.
 * @param attackSupportDetectorConfidenceThreshold [in] Attack support detector confidence threshold, in the range [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePad_SetAttackSupportDetectorConfidenceThreshold(ID3_FACE_PAD hFacePad, int attackSupportDetectorConfidenceThreshold);

/**
 * Computes a blurriness score for a color image.
 * An attack presented on a low resolution support has more chance to be blurred than a bonafide.
 * The maximum recommended value is 20.
 * Important: Loading the model ``FaceBlurrinessDetector`` is required to use this function.
 * @param hFacePad [in] Handle to the FacePad object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param score [out] The computed blurriness score, in the range [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePad_ComputeBlurrinessScore(ID3_FACE_PAD hFacePad, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, int *score);

/**
 * Computes the PAD score and confidence of a detected face using only the color image.
 * A high score means a high probability for the face to be bona-fide, hence not an attack. The minimum recommended score is 90.
 * A low confidence means that the quality of the image is not sufficient enough to take a decision. The minimum recommended confidence is 70.
 * Warning: A minimum IOD (64 pixels per default) for the detected face is required for this function, below this value it will output an error.
 * Important: Loading the ``FaceColorBasedPad`` model is required to use this function.
 * @param hFacePad [in] Handle to the FacePad object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param sColorBasedPadResult [out] The computed PAD result including score and confidence.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePad_ComputeColorBasedScore(ID3_FACE_PAD hFacePad, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, id3FaceColorBasedPadResult *sColorBasedPadResult);

/**
 * Computes the PAD score of a detected face using a depth map image.
 * A high score means a high probability for the face to be bona-fide, hence not an attack.
 * The minimum recommended value is 10.
 * Important: Loading the ``FaceDepthBasedPad`` model is required to use this function.
 * @param hFacePad [in] Handle to the FacePad object.
 * @param hImage [in] Source image to process. Must be Grayscale 16 Bits.
 * @param hDetectedFace [in] Detected face to process.
 * @param score [out] The computed PAD score, in the range [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePad_ComputeDepthBasedScore(ID3_FACE_PAD hFacePad, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, int *score);

/**
 * Detects if an attack support surrounds the detected face.
 * The recommended usage is to consider the image to be an attack as soon as an attack support is detected.
 * Important: Loading the ``FaceAttackSupportDetector`` model is required to use this function.
 * @param hFacePad [in] Handle to the FacePad object.
 * @param hImage [in] Source image to process.
 * @param hDetectedFace [in] Detected face to process.
 * @param sDetectedFaceAttackSupport [out] The detected face attack support.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePad_DetectAttackSupport(ID3_FACE_PAD hFacePad, ID3_FACE_IMAGE hImage, ID3_DETECTED_FACE hDetectedFace, id3DetectedFaceAttackSupport *sDetectedFaceAttackSupport);

#ifdef __cplusplus
}
#endif

#endif
