//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FacePortraitProcessor_h
#define _id3FacePortraitProcessor_h

#include <id3Face/id3FaceExports.h>
#include <id3Face/id3FaceModel.h>
#include <id3Face/id3FaceImage.h>
#include <id3Face/id3FacePortrait.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Facilitates the creation of portraits in various scenarios such as enrolment or face verification.
 */

/**
 * Handle to a PortraitProcessor object of the Face module.
 */
typedef struct id3FacePortraitProcessor *ID3_FACE_PORTRAIT_PROCESSOR;

/**
 * @brief Initializes a PortraitProcessor object.
 * @param phPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_Initialize(ID3_FACE_PORTRAIT_PROCESSOR *phPortraitProcessor);

/**
 * @brief Releases the memory allocated for a PortraitProcessor.
 * @param phPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_Dispose(ID3_FACE_PORTRAIT_PROCESSOR *phPortraitProcessor);

/**
 * @brief Checks if the handle object is valid.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_CheckHandle(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor);

/**
 * Gets the background color uniformity threshold.
 * If the score is above this value, the background color of the image is definitely uniform.
 * Hint: Default value is 80.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param backgroundColorUniformityThreshold [out] Background color uniformity threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetBackgroundColorUniformityThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *backgroundColorUniformityThreshold);

/**
 * Sets the background color uniformity threshold.
 * If the score is above this value, the background color of the image is definitely uniform.
 * Hint: Default value is 80.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param backgroundColorUniformityThreshold [in] Background color uniformity threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetBackgroundColorUniformityThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int backgroundColorUniformityThreshold);

/**
 * Gets the background structure uniformity threshold.
 * If the score is above this value, the background structure of the image is definitely uniform.
 * Hint: Default value is 80.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param backgroundStructureUniformityThreshold [out] Background structure uniformity threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetBackgroundStructureUniformityThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *backgroundStructureUniformityThreshold);

/**
 * Sets the background structure uniformity threshold.
 * If the score is above this value, the background structure of the image is definitely uniform.
 * Hint: Default value is 80.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param backgroundStructureUniformityThreshold [in] Background structure uniformity threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetBackgroundStructureUniformityThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int backgroundStructureUniformityThreshold);

/**
 * Gets the aspect ratio (height to width) of the cropped portrait image.
 * Hint: Default value is 4/3.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param cropAspectRatio [out] Aspect ratio (height to width) of the cropped portrait image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetCropAspectRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float *cropAspectRatio);

/**
 * Sets the aspect ratio (height to width) of the cropped portrait image.
 * Hint: Default value is 4/3.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param cropAspectRatio [in] Aspect ratio (height to width) of the cropped portrait image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetCropAspectRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float cropAspectRatio);

/**
 * Gets the ratio between the vertical eye position and the image height for ICAO portrait cropping. Must be in the range ]0;1[.
 * Hint: Default value is 0.45.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param cropEyePositionRatio [out] Ratio between the vertical eye position and the image height for ICAO portrait cropping. Must be in the range ]0;1[.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetCropEyePositionRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float *cropEyePositionRatio);

/**
 * Sets the ratio between the vertical eye position and the image height for ICAO portrait cropping. Must be in the range ]0;1[.
 * Hint: Default value is 0.45.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param cropEyePositionRatio [in] Ratio between the vertical eye position and the image height for ICAO portrait cropping. Must be in the range ]0;1[.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetCropEyePositionRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float cropEyePositionRatio);

/**
 * Gets the ratio between the interocular distance (IOD) and the image width for ICAO portrait cropping. Must be in the range ]0;1[.
 * Hint: Default value is 0.25.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param cropIodRatio [out] Ratio between the interocular distance (IOD) and the image width for ICAO portrait cropping. Must be in the range ]0;1[.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetCropIodRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float *cropIodRatio);

/**
 * Sets the ratio between the interocular distance (IOD) and the image width for ICAO portrait cropping. Must be in the range ]0;1[.
 * Hint: Default value is 0.25.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param cropIodRatio [in] Ratio between the interocular distance (IOD) and the image width for ICAO portrait cropping. Must be in the range ]0;1[.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetCropIodRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float cropIodRatio);

/**
 * Gets the relative margin around the face for basic portrait cropping.
 * Hint: Default value is 0.5.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param cropMargin [out] Relative margin around the face for basic portrait cropping.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetCropMargin(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float *cropMargin);

/**
 * Sets the relative margin around the face for basic portrait cropping.
 * Hint: Default value is 0.5.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param cropMargin [in] Relative margin around the face for basic portrait cropping.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetCropMargin(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float cropMargin);

/**
 * Gets the face dynamic range threshold.
 * Above this threshold, the dynamic range of the image in the face region is good enough.
 * Hint: Default value is 128.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param dynamicRangeThreshold [out] Face dynamic range threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetDynamicRangeThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *dynamicRangeThreshold);

/**
 * Sets the face dynamic range threshold.
 * Above this threshold, the dynamic range of the image in the face region is good enough.
 * Hint: Default value is 128.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param dynamicRangeThreshold [in] Face dynamic range threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetDynamicRangeThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int dynamicRangeThreshold);

/**
 * Gets the equal brightness threshold.
 * Above this value, the face is equally exposed.
 * Hint: Default value is 10.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param equalBrightnessThreshold [out] Equal brightness threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetEqualBrightnessThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *equalBrightnessThreshold);

/**
 * Sets the equal brightness threshold.
 * Above this value, the face is equally exposed.
 * Hint: Default value is 10.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param equalBrightnessThreshold [in] Equal brightness threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetEqualBrightnessThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int equalBrightnessThreshold);

/**
 * Gets the sensibility of the expression classifier.
 * The higher the value, the more sensitive the algorithm will be, meaning that it will be less likely to estimate neutral expression out of the detected face.
 * Hint: Default value is 60.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param expressionSensibility [out] Sensibility of the expression classifier.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetExpressionSensibility(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *expressionSensibility);

/**
 * Sets the sensibility of the expression classifier.
 * The higher the value, the more sensitive the algorithm will be, meaning that it will be less likely to estimate neutral expression out of the detected face.
 * Hint: Default value is 60.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param expressionSensibility [in] Sensibility of the expression classifier.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetExpressionSensibility(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int expressionSensibility);

/**
 * Gets the visible eye threshold.
 * If the score is above this value, the eye is probably occluded.
 * Hint: Default value 80.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param eyeOcclusionThreshold [out] Visible eye threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetEyeOcclusionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *eyeOcclusionThreshold);

/**
 * Sets the visible eye threshold.
 * If the score is above this value, the eye is probably occluded.
 * Hint: Default value 80.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param eyeOcclusionThreshold [in] Visible eye threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetEyeOcclusionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int eyeOcclusionThreshold);

/**
 * Gets the eye opening threshold.
 * If the score is above this value, the eyes are definitely open.
 * Hint: Default value is 5.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param eyeOpeningThreshold [out] Eye opening threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetEyeOpeningThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *eyeOpeningThreshold);

/**
 * Sets the eye opening threshold.
 * If the score is above this value, the eyes are definitely open.
 * Hint: Default value is 5.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param eyeOpeningThreshold [in] Eye opening threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetEyeOpeningThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int eyeOpeningThreshold);

/**
 * Gets the the size, in pixels, of the downscaled image used for face detection.
 * Hint: Default value is 256. Low values tend to ignore small faces in the input image, while higher values tend to take longer to process.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param faceDetectionImageSize [out] The size, in pixels, of the downscaled image used for face detection.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetFaceDetectionImageSize(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *faceDetectionImageSize);

/**
 * Sets the the size, in pixels, of the downscaled image used for face detection.
 * Hint: Default value is 256. Low values tend to ignore small faces in the input image, while higher values tend to take longer to process.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param faceDetectionImageSize [in] The size, in pixels, of the downscaled image used for face detection.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetFaceDetectionImageSize(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int faceDetectionImageSize);

/**
 * Gets the specifies the model used to detect and track faces.
 * Hint: Default value is ``FaceDetector4B``. Some better accuracy/speed balances can be found by choosing another model.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param eFaceDetectionModel [out] Specifies the model used to detect and track faces.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetFaceDetectionModel(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, id3FaceModel *eFaceDetectionModel);

/**
 * Sets the specifies the model used to detect and track faces.
 * Hint: Default value is ``FaceDetector4B``. Some better accuracy/speed balances can be found by choosing another model.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param eFaceDetectionModel [in] Specifies the model used to detect and track faces.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetFaceDetectionModel(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, id3FaceModel eFaceDetectionModel);

/**
 * Gets the specifies the face detection confidence threshold, in the range is [0;100].
 * Hint: Default value is 50. Setting a high threshold reduces false detections but may increase the number of undetected faces.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param faceDetectionThreshold [out] Specifies the face detection confidence threshold, in the range is [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetFaceDetectionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *faceDetectionThreshold);

/**
 * Sets the specifies the face detection confidence threshold, in the range is [0;100].
 * Hint: Default value is 50. Setting a high threshold reduces false detections but may increase the number of undetected faces.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param faceDetectionThreshold [in] Specifies the face detection confidence threshold, in the range is [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetFaceDetectionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int faceDetectionThreshold);

/**
 * Gets the model used to create features and assess consistancy among views of a given face.
 * Hint: Default value is ``FaceEncoder9B``. Some better accuracy/speed balances can be found by choosing another model.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param eFaceEncodingModel [out] Model used to create features and assess consistancy among views of a given face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetFaceEncodingModel(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, id3FaceModel *eFaceEncodingModel);

/**
 * Sets the model used to create features and assess consistancy among views of a given face.
 * Hint: Default value is ``FaceEncoder9B``. Some better accuracy/speed balances can be found by choosing another model.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param eFaceEncodingModel [in] Model used to create features and assess consistancy among views of a given face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetFaceEncodingModel(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, id3FaceModel eFaceEncodingModel);

/**
 * Gets the face mask detection threshold.
 * If the detection score is above this value, the person is probably wearing a mask.
 * Hint: Default value is 35.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param faceMaskThreshold [out] Face mask detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetFaceMaskThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *faceMaskThreshold);

/**
 * Sets the face mask detection threshold.
 * If the detection score is above this value, the person is probably wearing a mask.
 * Hint: Default value is 35.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param faceMaskThreshold [in] Face mask detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetFaceMaskThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int faceMaskThreshold);

/**
 * Gets the glasses detection threshold.
 * If the score is above this value, the person is probably wearing glasses.
 * Hint: Default value 90.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param glassesThreshold [out] Glasses detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetGlassesThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *glassesThreshold);

/**
 * Sets the glasses detection threshold.
 * If the score is above this value, the person is probably wearing glasses.
 * Hint: Default value 90.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param glassesThreshold [in] Glasses detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetGlassesThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int glassesThreshold);

/**
 * Gets the threshold for confirming that the person is wearing a hat.
 * Hint: Default value is 98.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hatThreshold [out] Threshold for confirming that the person is wearing a hat.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetHatThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *hatThreshold);

/**
 * Sets the threshold for confirming that the person is wearing a hat.
 * Hint: Default value is 98.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hatThreshold [in] Threshold for confirming that the person is wearing a hat.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetHatThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int hatThreshold);

/**
 * Gets the make-up detection threshold.
 * If the detection score is above this value, the persone is definitely wearing make-up.
 * Hint: Default value 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param makeupThreshold [out] Make-up detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMakeupThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *makeupThreshold);

/**
 * Sets the make-up detection threshold.
 * If the detection score is above this value, the persone is definitely wearing make-up.
 * Hint: Default value 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param makeupThreshold [in] Make-up detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMakeupThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int makeupThreshold);

/**
 * Gets the male gender detection threshold.
 * If the score is higher than this value, the person is probably male.
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maleThreshold [out] Male gender detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaleThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *maleThreshold);

/**
 * Sets the male gender detection threshold.
 * If the score is higher than this value, the person is probably male.
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maleThreshold [in] Male gender detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaleThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int maleThreshold);

/**
 * Gets the maximum value of the eye gaze alongside the X axis.
 * Hint: Default value is 10.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumEyeGazeX [out] Maximum value of the eye gaze alongside the X axis.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaximumEyeGazeX(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *maximumEyeGazeX);

/**
 * Sets the maximum value of the eye gaze alongside the X axis.
 * Hint: Default value is 10.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumEyeGazeX [in] Maximum value of the eye gaze alongside the X axis.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaximumEyeGazeX(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int maximumEyeGazeX);

/**
 * Gets the maximum value of the eye gaze alongside the Y axis.
 * Hint: Default value is 10.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumEyeGazeY [out] Maximum value of the eye gaze alongside the Y axis.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaximumEyeGazeY(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *maximumEyeGazeY);

/**
 * Sets the maximum value of the eye gaze alongside the Y axis.
 * Hint: Default value is 10.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumEyeGazeY [in] Maximum value of the eye gaze alongside the Y axis.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaximumEyeGazeY(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int maximumEyeGazeY);

/**
 * Gets the maximum distance between the base of the chin and the crown is less than 80% of the total height of the image.
 * For ICAO compliance, this distance must be in the range 60 to 90.
 * Hint: Default value is 90.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumHeadHeightRatio [out] Maximum distance between the base of the chin and the crown is less than 80% of the total height of the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaximumHeadHeightRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *maximumHeadHeightRatio);

/**
 * Sets the maximum distance between the base of the chin and the crown is less than 80% of the total height of the image.
 * For ICAO compliance, this distance must be in the range 60 to 90.
 * Hint: Default value is 90.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumHeadHeightRatio [in] Maximum distance between the base of the chin and the crown is less than 80% of the total height of the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaximumHeadHeightRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int maximumHeadHeightRatio);

/**
 * Gets the value indicating if the image width is conform to the (image width / head width) ratio of 7:5.
 * The head width is defined as the distance between the left and right ears.
 * For ICAO compliance, the value must be in the range 50 to 75.
 * Hint: Default value is 75.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumHeadWidthRatio [out] Value indicating if the image width is conform to the (image width / head width) ratio of 7:5.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaximumHeadWidthRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *maximumHeadWidthRatio);

/**
 * Sets the value indicating if the image width is conform to the (image width / head width) ratio of 7:5.
 * The head width is defined as the distance between the left and right ears.
 * For ICAO compliance, the value must be in the range 50 to 75.
 * Hint: Default value is 75.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumHeadWidthRatio [in] Value indicating if the image width is conform to the (image width / head width) ratio of 7:5.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaximumHeadWidthRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int maximumHeadWidthRatio);

/**
 * Gets the value indicating if the head is correctly centered horizontally.
 * For ICAO compliance, the value must be in the range 45 to 55.
 * Hint: Default value is 55.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumHorizontalPosition [out] Value indicating if the head is correctly centered horizontally.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaximumHorizontalPosition(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *maximumHorizontalPosition);

/**
 * Sets the value indicating if the head is correctly centered horizontally.
 * For ICAO compliance, the value must be in the range 45 to 55.
 * Hint: Default value is 55.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumHorizontalPosition [in] Value indicating if the head is correctly centered horizontally.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaximumHorizontalPosition(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int maximumHorizontalPosition);

/**
 * Gets the maximum interocular distance (IOD), in pixels.
 * This parameter is used to prevent the subject from being too close to the camera.
 * Hint: Default value is 512.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumIod [out] Maximum interocular distance (IOD), in pixels.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaximumIod(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *maximumIod);

/**
 * Sets the maximum interocular distance (IOD), in pixels.
 * This parameter is used to prevent the subject from being too close to the camera.
 * Hint: Default value is 512.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumIod [in] Maximum interocular distance (IOD), in pixels.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaximumIod(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int maximumIod);

/**
 * Gets the maximum head pitch angle, in degrees, for frontal positioning.
 * Hint: Default value is 20.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumPitchAngle [out] Maximum head pitch angle, in degrees, for frontal positioning.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaximumPitchAngle(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float *maximumPitchAngle);

/**
 * Sets the maximum head pitch angle, in degrees, for frontal positioning.
 * Hint: Default value is 20.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumPitchAngle [in] Maximum head pitch angle, in degrees, for frontal positioning.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaximumPitchAngle(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float maximumPitchAngle);

/**
 * Gets the maximum head roll angle, in degrees, for frontal positioning.
 * Hint: Default value is 15.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumRollAngle [out] Maximum head roll angle, in degrees, for frontal positioning.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaximumRollAngle(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float *maximumRollAngle);

/**
 * Sets the maximum head roll angle, in degrees, for frontal positioning.
 * Hint: Default value is 15.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumRollAngle [in] Maximum head roll angle, in degrees, for frontal positioning.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaximumRollAngle(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float maximumRollAngle);

/**
 * Gets the maximum distance from the bottom edge of the image to the imaginary line passing through the center of the eyes is between 50% - 70% of the total vertical length of the image.
 * For ICAO compliance, the value must be in the range 30 to 50.
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumVerticalPosition [out] Maximum distance from the bottom edge of the image to the imaginary line passing through the center of the eyes is between 50% - 70% of the total vertical length of the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaximumVerticalPosition(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *maximumVerticalPosition);

/**
 * Sets the maximum distance from the bottom edge of the image to the imaginary line passing through the center of the eyes is between 50% - 70% of the total vertical length of the image.
 * For ICAO compliance, the value must be in the range 30 to 50.
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumVerticalPosition [in] Maximum distance from the bottom edge of the image to the imaginary line passing through the center of the eyes is between 50% - 70% of the total vertical length of the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaximumVerticalPosition(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int maximumVerticalPosition);

/**
 * Gets the maximum head yaw angle, in degrees, for frontal positioning.
 * Hint: Default value is 8.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumYawAngle [out] Maximum head yaw angle, in degrees, for frontal positioning.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMaximumYawAngle(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float *maximumYawAngle);

/**
 * Sets the maximum head yaw angle, in degrees, for frontal positioning.
 * Hint: Default value is 8.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param maximumYawAngle [in] Maximum head yaw angle, in degrees, for frontal positioning.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMaximumYawAngle(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, float maximumYawAngle);

/**
 * Gets the minimum distance between the base of the chin and the crown is less than 80% of the total height of the image.
 * For ICAO compliance, this distance must be in the range 60 to 90.
 * Hint: Default value is 60.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param minimumHeadHeightRatio [out] Minimum distance between the base of the chin and the crown is less than 80% of the total height of the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMinimumHeadHeightRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *minimumHeadHeightRatio);

/**
 * Sets the minimum distance between the base of the chin and the crown is less than 80% of the total height of the image.
 * For ICAO compliance, this distance must be in the range 60 to 90.
 * Hint: Default value is 60.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param minimumHeadHeightRatio [in] Minimum distance between the base of the chin and the crown is less than 80% of the total height of the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMinimumHeadHeightRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int minimumHeadHeightRatio);

/**
 * Gets the value indicating if the image width is conform to the (image width / head width) ratio of 7:5.
 * The head width is defined as the distance between the left and right ears.
 * For ICAO compliance, the value must be in the range 50 to 75.
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param minimumHeadWidthRatio [out] Value indicating if the image width is conform to the (image width / head width) ratio of 7:5.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMinimumHeadWidthRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *minimumHeadWidthRatio);

/**
 * Sets the value indicating if the image width is conform to the (image width / head width) ratio of 7:5.
 * The head width is defined as the distance between the left and right ears.
 * For ICAO compliance, the value must be in the range 50 to 75.
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param minimumHeadWidthRatio [in] Value indicating if the image width is conform to the (image width / head width) ratio of 7:5.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMinimumHeadWidthRatio(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int minimumHeadWidthRatio);

/**
 * Gets the value indicating if the head is correctly centered horizontally.
 * For ICAO compliance, the value must be in the range 45 to 55.
 * Hint: Default value is 45.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param minimumHorizontalPosition [out] Value indicating if the head is correctly centered horizontally.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMinimumHorizontalPosition(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *minimumHorizontalPosition);

/**
 * Sets the value indicating if the head is correctly centered horizontally.
 * For ICAO compliance, the value must be in the range 45 to 55.
 * Hint: Default value is 45.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param minimumHorizontalPosition [in] Value indicating if the head is correctly centered horizontally.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMinimumHorizontalPosition(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int minimumHorizontalPosition);

/**
 * Gets the minimum intra-occular distance, in pixels.
 * This parameter is used to the limit the distance from the subject to the camera.
 * Hint: Default value is 0.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param minimumIod [out] Minimum intra-occular distance, in pixels.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMinimumIod(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *minimumIod);

/**
 * Sets the minimum intra-occular distance, in pixels.
 * This parameter is used to the limit the distance from the subject to the camera.
 * Hint: Default value is 0.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param minimumIod [in] Minimum intra-occular distance, in pixels.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMinimumIod(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int minimumIod);

/**
 * Gets the minimum distance from the bottom edge of the image to the imaginary line passing through the center of the eyes is between 50% - 70% of the total vertical length of the image.
 * For ICAO compliance, the value must be in the range 30 to 50.
 * Hint: Default value is 30.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param minimumVerticalPosition [out] Minimum distance from the bottom edge of the image to the imaginary line passing through the center of the eyes is between 50% - 70% of the total vertical length of the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMinimumVerticalPosition(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *minimumVerticalPosition);

/**
 * Sets the minimum distance from the bottom edge of the image to the imaginary line passing through the center of the eyes is between 50% - 70% of the total vertical length of the image.
 * For ICAO compliance, the value must be in the range 30 to 50.
 * Hint: Default value is 30.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param minimumVerticalPosition [in] Minimum distance from the bottom edge of the image to the imaginary line passing through the center of the eyes is between 50% - 70% of the total vertical length of the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMinimumVerticalPosition(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int minimumVerticalPosition);

/**
 * Gets the mouth occlusion threshold. 
 * If the mouth occlusion score is above this value, the mouth is probably occluded.
 * Hint: Default value 70.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param mouthOcclusionThreshold [out] Mouth occlusion threshold. 
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMouthOcclusionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *mouthOcclusionThreshold);

/**
 * Sets the mouth occlusion threshold. 
 * If the mouth occlusion score is above this value, the mouth is probably occluded.
 * Hint: Default value 70.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param mouthOcclusionThreshold [in] Mouth occlusion threshold. 
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMouthOcclusionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int mouthOcclusionThreshold);

/**
 * Gets the mouth opening threshold.
 * If the mouth opening score is above this value, the mouth of detected face is definitely open.
 * Hint: Default value is 35.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param mouthOpenThreshold [out] Mouth opening threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetMouthOpenThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *mouthOpenThreshold);

/**
 * Sets the mouth opening threshold.
 * If the mouth opening score is above this value, the mouth of detected face is definitely open.
 * Hint: Default value is 35.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param mouthOpenThreshold [in] Mouth opening threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetMouthOpenThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int mouthOpenThreshold);

/**
 * Gets the natural skin color threshold.
 * Above this threshold, th skin of the person looks natural.
 * Hint: Default value is 5.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param naturalSkinColorThreshold [out] Natural skin color threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetNaturalSkinColorThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *naturalSkinColorThreshold);

/**
 * Sets the natural skin color threshold.
 * Above this threshold, th skin of the person looks natural.
 * Hint: Default value is 5.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param naturalSkinColorThreshold [in] Natural skin color threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetNaturalSkinColorThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int naturalSkinColorThreshold);

/**
 * Gets the flash reflexion detection threshold.
 * Above this threshold, the flash reflexion is probably present on the face.
 * Hint: Default value is 70.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param noFlashReflectionThreshold [out] Flash reflexion detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetNoFlashReflectionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *noFlashReflectionThreshold);

/**
 * Sets the flash reflexion detection threshold.
 * Above this threshold, the flash reflexion is probably present on the face.
 * Hint: Default value is 70.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param noFlashReflectionThreshold [in] Flash reflexion detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetNoFlashReflectionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int noFlashReflectionThreshold);

/**
 * Gets the threshold for confirming the presence of noise in the image.
 * Hint: Default value is 90.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param noiseThreshold [out] Threshold for confirming the presence of noise in the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetNoiseThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *noiseThreshold);

/**
 * Sets the threshold for confirming the presence of noise in the image.
 * Hint: Default value is 90.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param noiseThreshold [in] Threshold for confirming the presence of noise in the image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetNoiseThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int noiseThreshold);

/**
 * Gets the nose occlusion threshold. 
 * If the score is above this value, the nose is probably occluded.
 * Hint: Default value 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param noseOcclusionThreshold [out] Nose occlusion threshold. 
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetNoseOcclusionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *noseOcclusionThreshold);

/**
 * Sets the nose occlusion threshold. 
 * If the score is above this value, the nose is probably occluded.
 * Hint: Default value 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param noseOcclusionThreshold [in] Nose occlusion threshold. 
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetNoseOcclusionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int noseOcclusionThreshold);

/**
 * Gets the over exposure threshold.
 * Above this threshold, the ratio of face pixels are over-exposed. 
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param overExposureThreshold [out] Over exposure threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetOverExposureThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *overExposureThreshold);

/**
 * Sets the over exposure threshold.
 * Above this threshold, the ratio of face pixels are over-exposed. 
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param overExposureThreshold [in] Over exposure threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetOverExposureThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int overExposureThreshold);

/**
 * Gets the presentation attack detection threshold, from 0 to 100.
 * Higher values enable more attacks to be detected, but may reduce ease of use.
 * Hint: Default value 95.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param padThreshold [out] Presentation attack detection threshold, from 0 to 100.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetPadThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *padThreshold);

/**
 * Sets the presentation attack detection threshold, from 0 to 100.
 * Higher values enable more attacks to be detected, but may reduce ease of use.
 * Hint: Default value 95.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param padThreshold [in] Presentation attack detection threshold, from 0 to 100.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetPadThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int padThreshold);

/**
 * Gets the pixelation detection threshold.
 * Above this threshold, the image is probably pixelated.
 * Hint: Default value is 90.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param pixelationThreshold [out] Pixelation detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetPixelationThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *pixelationThreshold);

/**
 * Sets the pixelation detection threshold.
 * Above this threshold, the image is probably pixelated.
 * Hint: Default value is 90.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param pixelationThreshold [in] Pixelation detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetPixelationThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int pixelationThreshold);

/**
 * Gets the threshold for confirming the presence of red eyes.
 * Hint: Default value is 10.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param redEyeThreshold [out] Threshold for confirming the presence of red eyes.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetRedEyeThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *redEyeThreshold);

/**
 * Sets the threshold for confirming the presence of red eyes.
 * Hint: Default value is 10.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param redEyeThreshold [in] Threshold for confirming the presence of red eyes.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetRedEyeThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int redEyeThreshold);

/**
 * Gets the threshold indicating whether the image resolution is sufficient.
 * Hint: Default value is 90.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param resolutionThreshold [out] Threshold indicating whether the image resolution is sufficient.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetResolutionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *resolutionThreshold);

/**
 * Sets the threshold indicating whether the image resolution is sufficient.
 * Hint: Default value is 90.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param resolutionThreshold [in] Threshold indicating whether the image resolution is sufficient.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetResolutionThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int resolutionThreshold);

/**
 * Gets the sharpness threshold. 
 * If the sharpness score is above this value, the image sharpness is sufficient.
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param sharpnessThreshold [out] Sharpness threshold. 
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetSharpnessThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *sharpnessThreshold);

/**
 * Sets the sharpness threshold. 
 * If the sharpness score is above this value, the image sharpness is sufficient.
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param sharpnessThreshold [in] Sharpness threshold. 
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetSharpnessThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int sharpnessThreshold);

/**
 * Gets the smile threshold.
 * If the Smile score is above this value, the detected face is definitely smiling.
 * Hint: Default value is 75.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param smileThreshold [out] Smile threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetSmileThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *smileThreshold);

/**
 * Sets the smile threshold.
 * If the Smile score is above this value, the detected face is definitely smiling.
 * Hint: Default value is 75.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param smileThreshold [in] Smile threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetSmileThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int smileThreshold);

/**
 * Gets the number of threads to be used for portrait processing.
 * Hint: Default value is 1. Allocating more than one thread can increase the speed of the process.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param threadCount [out] Number of threads to be used for portrait processing.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetThreadCount(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *threadCount);

/**
 * Sets the number of threads to be used for portrait processing.
 * Hint: Default value is 1. Allocating more than one thread can increase the speed of the process.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param threadCount [in] Number of threads to be used for portrait processing.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetThreadCount(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int threadCount);

/**
 * Gets the tinted glass detection threshold.
 * If the score is above this value, the person is definitely wearing tinted glasses.
 * Hint: Default value 25.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param tintedGlassThreshold [out] Tinted glass detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetTintedGlassThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *tintedGlassThreshold);

/**
 * Sets the tinted glass detection threshold.
 * If the score is above this value, the person is definitely wearing tinted glasses.
 * Hint: Default value 25.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param tintedGlassThreshold [in] Tinted glass detection threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetTintedGlassThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int tintedGlassThreshold);

/**
 * Gets the minimum match score to reach to preserve the ID of a tracked face between frame 't-1' and frame 't'.
 * Hint: Default value is 3000 which corresponds to a False Match Rate of 1/1000.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param trackerMatchingThreshold [out] Minimum match score to reach to preserve the ID of a tracked face between frame 't-1' and frame 't'.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetTrackerMatchingThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *trackerMatchingThreshold);

/**
 * Sets the minimum match score to reach to preserve the ID of a tracked face between frame 't-1' and frame 't'.
 * Hint: Default value is 3000 which corresponds to a False Match Rate of 1/1000.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param trackerMatchingThreshold [in] Minimum match score to reach to preserve the ID of a tracked face between frame 't-1' and frame 't'.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetTrackerMatchingThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int trackerMatchingThreshold);

/**
 * Gets the maximum number of consecutive non-detections to be reached before deleting a tracked face.
 * Hint: Default value is 2. One must adapt this value to its needs in terms of tracker identity memory (in seconds) and measured frame rate on target platform.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param trackerMaximumFaceAge [out] Maximum number of consecutive non-detections to be reached before deleting a tracked face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetTrackerMaximumFaceAge(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *trackerMaximumFaceAge);

/**
 * Sets the maximum number of consecutive non-detections to be reached before deleting a tracked face.
 * Hint: Default value is 2. One must adapt this value to its needs in terms of tracker identity memory (in seconds) and measured frame rate on target platform.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param trackerMaximumFaceAge [in] Maximum number of consecutive non-detections to be reached before deleting a tracked face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetTrackerMaximumFaceAge(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int trackerMaximumFaceAge);

/**
 * Gets the minimum number of consecutive detections to reach before creating a tracked face.
 * Hint: Default value is 1 for ``FaceDetector4B`` since the false detection rate is low enough. If using a less accurate detector (such as FaceDetector3C) one might consider increasing a bit this value to avoid false tracks. 
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param trackerMinimumFaceAge [out] Minimum number of consecutive detections to reach before creating a tracked face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetTrackerMinimumFaceAge(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *trackerMinimumFaceAge);

/**
 * Sets the minimum number of consecutive detections to reach before creating a tracked face.
 * Hint: Default value is 1 for ``FaceDetector4B`` since the false detection rate is low enough. If using a less accurate detector (such as FaceDetector3C) one might consider increasing a bit this value to avoid false tracks. 
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param trackerMinimumFaceAge [in] Minimum number of consecutive detections to reach before creating a tracked face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetTrackerMinimumFaceAge(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int trackerMinimumFaceAge);

/**
 * Gets the non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range [0;100].
 * Hint: Default value is 0. Setting a high threshold allows to detect more overlapping faces which can be useful in a multi-face scenario. On the contrary, in a portrait scenario, a low NMS IOU threshold should be preferred.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param trackerNmsIouThreshold [out] Non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetTrackerNmsIouThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *trackerNmsIouThreshold);

/**
 * Sets the non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range [0;100].
 * Hint: Default value is 0. Setting a high threshold allows to detect more overlapping faces which can be useful in a multi-face scenario. On the contrary, in a portrait scenario, a low NMS IOU threshold should be preferred.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param trackerNmsIouThreshold [in] Non-maximum suppression (NMS) intersection-over-union (IOU) threshold, in the range [0;100].
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetTrackerNmsIouThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int trackerNmsIouThreshold);

/**
 * Gets the under exposure threshold.
 * Above this threshold, the face is probably under-exposed.
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param underExposureThreshold [out] Under exposure threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_GetUnderExposureThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int *underExposureThreshold);

/**
 * Sets the under exposure threshold.
 * Above this threshold, the face is probably under-exposed.
 * Hint: Default value is 50.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param underExposureThreshold [in] Under exposure threshold.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_SetUnderExposureThreshold(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, int underExposureThreshold);

/**
 * Analyses an image and creates a portrait with the first detected face or the currently tracked face if the update parameter is activated.
 * This method performs the following tasks:
 * - Face detection
 * - Landmark detection (68 points)
 * - Head pose estimation
 * - Verification of subject position
 * - Geometric attributes computation
 * - Unified quality score computation
 * Important: This method requires the ``FaceDetector``, ``FaceLandmarksEstimator`` and ``FacePoseEstimator`` models to be loaded.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hImage [in] Source image.
 * @param hPortrait [out] The newly created portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_CreatePortrait(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_IMAGE hImage, ID3_FACE_PORTRAIT hPortrait);

/**
 * Creates an ICAO-compliant portrait image by cropping the source image around the detected face.
 * This method uses the following parameters:
 * - Portrait aspect ratio
 * - Portrait eye position ratio
 * - Portrait IOD ratio
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] Source portrait.
 * @param hImage [out] The output cropped image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_CropIcaoPortrait(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait, ID3_FACE_IMAGE hImage);

/**
 * Creates a portrait image by cropping the source image around the detected face.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] The portrait to extract the face from.
 * @param hImage [out] The output cropped image of the face.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_CropPortrait(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait, ID3_FACE_IMAGE hImage);

/**
 * Detects occlusions on the subject's face.
 * Important: This function requires the ``FaceOcclusionDetector2A`` model to be loaded.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] The portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_DetectOcclusions(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait);

/**
 * Detects a possible presentation attack.
 * Important: This method requires the ``FaceColorBasedPad`` models to be loaded.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] The portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_DetectPresentationAttack(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait);

/**
 * Estimates the age of the person.
 * Important: This method requires the ``FaceAgeEstimator1A`` model to be loaded.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] The portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_EstimateAge(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait);

/**
 * Checks the background uniformity of the specified portrait.
 * Important: This method requires the ``FaceBackgroundUniformityEstimator1A`` model to be loaded.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] The portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_EstimateBackgroundUniformity(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait);

/**
 * Estimates the subject's expression on the specified portrait.
 * Important: This method requires the ``FaceExpressionClassifier1A`` model to be loaded.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] The portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_EstimateExpression(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait);

/**
 * Estimates the face attributes of the specified portrait.
 * It includes glasses, hat, makeup, mouth opening, gender, smile, eye gaze, eye opening, face mask
 * Note: The appropriate AI models must be loaded.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] The portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_EstimateFaceAttributes(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait);

/**
 * Estimates the geometric quality of the specified portrait according to the ICAO portrait parameters.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] The portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_EstimateGeometryQuality(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait);

/**
 * Etimates the photographic quality of the specified portrait.
 * It includes image colorization, equal brightness, dynamic range, resolution, natural skin color, flash reflection, noise, over-exposure, under-exposure, pixelation, sharpness.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] The portrait.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_EstimatePhotographicQuality(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait);

/**
 * Removes the background on the specified portrait image.
 * Important: This method requires the ``FaceBackgroundSegmenter1A`` model to be loaded.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hImage [in] The portrait image.
 * @param red [in] Green channel of the background color. Must be a value from 0 to 255.
 * @param green [in] Red channel of the background color. Must be a value from 0 to 255.
 * @param blue [in] Blue channel of the background color. Must be a value from 0 to 255.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_RemoveBackground(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_IMAGE hImage, int red, int green, int blue);

/**
 * Resets the face tracker.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_Reset(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor);

/**
 * Analyses an image and updates the specified portrait with the first detected face or the currently tracked face.
 * This method performs the following tasks:
 * - Face detection and tracking with prediction models
 * - Landmark detection (68 points)
 * - Head pose estimation
 * - Verification of subject position
 * - Template encoding and updating
 * - Geometric attributes computation
 * - Unified quality score computation
 * Important: This method requires the ``FaceDetector``, ``FaceEncoder``, ``FaceLandmarksEstimator`` and ``FacePoseEstimator`` models to be loaded.
 * @param hPortraitProcessor [in] Handle to the PortraitProcessor object.
 * @param hPortrait [in] The portrait to be updated.
 * @param hImage [in] The source image.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FacePortraitProcessor_UpdatePortrait(ID3_FACE_PORTRAIT_PROCESSOR hPortraitProcessor, ID3_FACE_PORTRAIT hPortrait, ID3_FACE_IMAGE hImage);

#ifdef __cplusplus
}
#endif

#endif
