//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3FaceStringList_h
#define _id3FaceStringList_h

#include <id3Face/id3FaceExports.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a list of strings.
 */

/**
 * Handle to a StringList object of the Face module.
 */
typedef struct id3FaceStringList *ID3_FACE_STRING_LIST;

/**
 * @brief Initializes a StringList object.
 * @param phStringList [in] Handle to the StringList object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceStringList_Initialize(ID3_FACE_STRING_LIST *phStringList);

/**
 * @brief Releases the memory allocated for a StringList.
 * @param phStringList [in] Handle to the StringList object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceStringList_Dispose(ID3_FACE_STRING_LIST *phStringList);

/**
 * @brief Checks if the handle object is valid.
 * @param hStringList [in] Handle to the StringList object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceStringList_CheckHandle(ID3_FACE_STRING_LIST hStringList);

/**
 * Clears the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceStringList_Clear(ID3_FACE_STRING_LIST hStringList);

/**
 * Adds an item to the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param item [in]  item to add.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceStringList_Add(ID3_FACE_STRING_LIST hStringList, const char *item);

/**
 * Gets an item of the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param index [in] Index of the  item to get.
 * @param item [out]  item to get.
 * @param itemSize [in,out] Size of the 'item' buffer. Receives the minimum required buffer size to hold the 'item' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'item' (excluding terminating NULL character).
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceStringList_Get(ID3_FACE_STRING_LIST hStringList, int index, char *item, int *itemSize);

/**
 * Sets an item of the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param index [in] Index of the  item to set.
 * @param item [in]  item to set.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceStringList_Set(ID3_FACE_STRING_LIST hStringList, int index, const char *item);

/**
 * Gets the number of elements in the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param count [out] Number of elements in the  object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceStringList_GetCount(ID3_FACE_STRING_LIST hStringList, int *count);

/**
 * Removes an element of the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param index [in] Index of the  item to remove.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceStringList_RemoveAt(ID3_FACE_STRING_LIST hStringList, int index);

/**
 * StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param newSize [in]  object.
 * @return An error code. See id3FaceError file for the list of possible error codes.
 * @retval id3FaceError_Success The function succeeded.
 */
ID3FACE_C_API int ID3_CDECL id3FaceStringList_Resize(ID3_FACE_STRING_LIST hStringList, int newSize);

#ifdef __cplusplus
}
#endif

#endif
